#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using System.Threading;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion

namespace gov.va.med.vbecs.BOL.UnitTests
{
	[TestFixture]
	public class BOL_PatientUnitSelection : BaseTest 
	{
		// The two objects below are the primary objects used for testing this class
		private BOL.PatientUnitSelection _patientUnitSelection;
		private BOL.OrderedComponent _orderedComponent;

		// Sometimes, there are unusual conditions and supplementary objects have to be created
		private BOL.PatientUnitSelection _patientUnitSelectionTwo;
		private BOL.OrderedComponent _orderedComponentTwo;

		// Use this for only simple tests
		private BOL.BloodUnit _genericBU;

		private string _cleanupSQL = string.Empty;
		private string _cleanupSQLTwo = string.Empty;

		// This is the specimen that is created first and not the one associated
		// with the PatientUnitSelecion object that we're testing with
		Guid _specimenOneGuid = Guid.Empty;

		// OrderedTest associated with the specimen tests that are associated with
		// the specimen that is associated with the PUS object.  Got that?
		Guid _orderedTestGuid = Guid.Empty;
		
		/// <summary>
		/// This method is used to set up the conditions for the tests.  In order for the tests to run, there must
		/// be at least one OrderedComponent in the DataBase that has an associated PatientSpecimenGuid
		/// </summary>
		[SetUp]
		protected void SetUp()
		{
			this.BaseSetUp();

			if(this.RefreshData)
			{
				Guid patientGuid = CreatePatient();

				SetUpData(patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true);
			}
		}

		[TearDown]
		protected void TearDown()
		{
			this.ClearTestData();
		}


		#region constructors

		[Test]
		public void Ctor_Pass()
		{
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitSelection != null);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void Ctor_Fail()
		{
			_patientUnitSelection = new BOL.PatientUnitSelection(null, this.GetVbecsUser(Common.UserRoleType.Technologist));
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void Ctor_Fail_2()
		{
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, null);
		}


		#endregion


		#region properties

		[Test]
		public void IsEmergencyIssue_Pass()
		{
			Assert.IsTrue(!_patientUnitSelection.IsEmergencyIssue);
		}
		[Test]
		public void IsEmergencyIssue_Fail()
		{
			_orderedComponent.AssociateWithSpecimen(null);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitSelection.IsEmergencyIssue);
		}

		[Test]
		public void EmergencyIssueReason_Pass()
		{
			Assert.IsTrue(_patientUnitSelection.EmergencyIssueReason.Trim().Length == 0);
		}
		[Test]
		public void EmergencyIssueReason_Fail()
		{
			_orderedComponent.AssociateWithSpecimen(null);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, GetVbecsUser(Common.UserRoleType.Technologist));
			Assert.IsTrue(_patientUnitSelection.EmergencyIssueReason.Trim().Length > 0);
		}


		[Test]
		public void OrderedComponent_Pass()
		{
			_patientUnitSelection.OrderedComponent = _orderedComponent;
			Assert.AreEqual(_patientUnitSelection.OrderedComponent, _orderedComponent);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void OrderedComponent_Fail()
		{
		}


		[Test]
		public void SelectedUnitsCollection_Pass()
		{
			gov.va.med.vbecs.BOL.PatientUnitSelection.SelectedUnits _SelectedUnitsCollection;
			_SelectedUnitsCollection = this._patientUnitSelection.SelectedUnitsCollection;
			Assert.AreEqual(this._patientUnitSelection.SelectedUnitsCollection,_SelectedUnitsCollection);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SelectedUnitsCollection_Fail()
		{
		}


		[Test]
		public void DirectedBloodUnits_Pass()
		{
			DataTable _DirectedBloodUnits;
			_DirectedBloodUnits = _patientUnitSelection.DirectedBloodUnits;
			Assert.AreEqual(_patientUnitSelection.DirectedBloodUnits,_DirectedBloodUnits);
		}
		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void DirectedBloodUnits_Fail()
		{
		}


		[Test]
		public void ValidationMessages_Get_Set()
		{
			System.Collections.ArrayList _validationMessages;
			_validationMessages = _patientUnitSelection.ValidationMessages;
			Assert.AreEqual(_patientUnitSelection.ValidationMessages,_validationMessages);

			ArrayList foo = new ArrayList();
			_patientUnitSelection.ValidationMessages = foo;
			Assert.AreEqual(_patientUnitSelection.ValidationMessages, foo);
		}
		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void ValidationMessages_Get_Fail()
		{
		}


		[Test]
		public void OrderedComponentComments_Pass()
		{
			System.Collections.ArrayList tstValue = new System.Collections.ArrayList();
			
			_patientUnitSelection.OrderedComponentComments = tstValue;

			Assert.AreEqual(this._patientUnitSelection.OrderedComponentComments, tstValue);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void OrderedComponentComments_Fail()
		{
		}


		[Test]
		public void SpecimenIdentified_Pass()
		{
			_patientUnitSelection.SpecimenIdentified = true;
			Assert.IsTrue(_patientUnitSelection.SpecimenIdentified);
		}
		[Test]
		public void SpecimenIdentified_Fail()
		{
			_patientUnitSelection.SpecimenIdentified = false;
			Assert.IsTrue(!_patientUnitSelection.SpecimenIdentified);
		}

		[Test]
		public void AlreadyOrderedUnits_Get()
		{
			DataTable _AlreadyOrderedUnits;
			_AlreadyOrderedUnits = this._patientUnitSelection.AlreadyOrderedUnits;
			Assert.AreEqual(this._patientUnitSelection.AlreadyOrderedUnits,_AlreadyOrderedUnits);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void AlreadyOrderedUnits_Get_Fail()
		{
		}

		#endregion


		#region methods

		[Test]
		public void GetEmergencyIssueStatus_305_Pass()
		{
			bool status = _patientUnitSelection.GetEmergencyIssueStatus();
			Assert.IsTrue(!status);
		}

		[Test]
		public void GetEmergencyIssueStatus_305_Fail()
		{
			bool status = false;
			
			// FIRST, test for the condition of no results
			_orderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;
			status = _patientUnitSelection.GetEmergencyIssueStatus();
			Assert.IsTrue(status && _patientUnitSelection.EmergencyIssueReason == Common.StrRes.ValidMsg.UC003.BR305NoABORhResults().ResString);
			// Reset
			_orderedComponent.Patient.AboRh.RH = Common.RH.Positive;




			// SECOND, test for no valid specimen by making the specimen unacceptable
			_orderedComponent.GetSpecimen().UnacceptableSpecimenIndicator = true;
			status = _patientUnitSelection.GetEmergencyIssueStatus();

			// Deleting these before running the assertion will ensure that data is correctly deleted
			// in the case if a unit test failure.Lohse moving below..
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE PatientSpecimenGuid = '" + _specimenOneGuid + "'");

			Assert.IsTrue(status && _patientUnitSelection.EmergencyIssueReason == Common.StrRes.ValidMsg.UC003.BR301Specimen().ResString, "No valid specimen");
			_orderedComponent.GetSpecimen().UnacceptableSpecimenIndicator = false;



			
			// THIRD, setup a result (repeat test) that conflicts with an existing result
			//Thread.Sleep(70000);	//Sleep allows the testdate to be incremented to avoid fnGetAboRh from puking
			// Setup a conflicting result
			Guid specimenTestGuid = SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "118", "B");
			Guid specimenTestTwoGuid = SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "121", "P");

			// refresh for cache data
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));
			status = _patientUnitSelection.GetEmergencyIssueStatus();
		
			// Deleting these before running the assertion will ensure that data is correctly deleted
			// in the case if a unit test failure.
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + specimenTestGuid + "'");
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + specimenTestTwoGuid + "'");

			Assert.IsTrue(status && _patientUnitSelection.EmergencyIssueReason == Common.StrRes.ValidMsg.UC003.BR340SpecimenResults().ResString, "Repeat specimen testing not consistent");
            



			// FOURTH, setup a result (different specimen) that conflicts with an existing result
			//Thread.Sleep(70000);	//Sleep allows the testdate to be incremented to avoid fnGetAboRh from puking
			// Setup a conflicting result
			specimenTestGuid = SetupSpecimenTest(_specimenOneGuid, _orderedTestGuid, "18", "B");
			specimenTestTwoGuid = SetupSpecimenTest(_specimenOneGuid, _orderedTestGuid, "21", "N");

			// refresh for cache data
			_orderedComponent.Patient = new BOL.Patient(_orderedComponent.Patient.PatientGuid);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent,  this.GetVbecsUser(Common.UserRoleType.Technologist));

			status = _patientUnitSelection.GetEmergencyIssueStatus();

			// Deleting these before running the assertion will ensure that data is correctly deleted
			// in the case if a unit test failure.
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE PatientSpecimenGuid = '" + _orderedComponent.GetSpecimen().SpecimenGuid + "'");
			Assert.IsTrue(status && _patientUnitSelection.EmergencyIssueReason == Common.StrRes.ValidMsg.UC003.BR330SpecimenResults().ResString, "Specimen test results not consistent");
			



			// FIFTH, set the ABO/Rh indicator to true
			UnitTestUtil.RunSQL("UPDATE Patient SET AboRhChangeIndicator = 1 WHERE PatientGuid = '" + _orderedComponent.Patient.PatientGuid + "'");
			
			// must force a retrieval from DB to get update patient information
			_orderedComponent.Patient = new BOL.Patient(_orderedComponent.Patient.PatientGuid);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));
			status = _patientUnitSelection.GetEmergencyIssueStatus();
			
			// Deleting these before running the assertion will ensure that data is correctly deleted
			// in the case if a unit test failure.
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + specimenTestGuid + "'");
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + specimenTestTwoGuid + "'");

			Assert.IsTrue(status && _patientUnitSelection.EmergencyIssueReason == Common.StrRes.ValidMsg.UC003.BR324ABORhConfirmation(string.Empty).ResString, "Testing not complete when justified ABO/Rh exists");
			UnitTestUtil.RunSQL("UPDATE Patient SET AboRhChangeIndicator = 0 WHERE PatientGuid = '" + _orderedComponent.Patient.PatientGuid + "'");
		}


		[Test]
		public void InSameDivision_Pass()
		{
			_genericBU.DivisionCode = _orderedComponent.DivisionCode;
			Assert.IsTrue(_patientUnitSelection.InSameDivision(_genericBU));
		}
		[Test]
		public void InSameDivision_Fail()
		{
			_genericBU.DivisionCode = "XXXXX";
			Assert.IsTrue(!_patientUnitSelection.InSameDivision(_genericBU));
		}

		[Test]
		public void IsOnPendingOutgoingShipmentList_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Assert.IsTrue(!_patientUnitSelection.IsOnPendingOutgoingShipmentList(bloodUnit));
		}
		[Test]
		public void IsOnPendingOutgoingShipmentList_Fail()
		{
			Guid lsGuid = DataCreator.GetRecordGuid("SELECT LocalSupplierGuid FROM VamcDivision WHERE DivisionCode = '" + this.TestDivisionCode + "'");
			BOL.BloodProduct tmpBP = new BOL.BloodProduct(DataCreator.GetRecordGuid("SELECT BloodProductGuid FROM BloodProduct WHERE ProductCode = 'E0291'"));
			BOL.BloodUnit tmpBU = CreateBloodUnit(tmpBP.ProductType.ProductTypeCode, tmpBP.ProductCode, Common.ABO.A, Common.RH.Positive, COMMON.ComponentClass.RBC);

			BOL.OutgoingShipmentCollection tmpOSC = BOL.OutgoingShipmentCollection.CreateOutgoingInvoice("UT7175-" + new Random().Next(999).ToString(), lsGuid, System.DateTime.Now.AddDays(-1), Common.ShipmentType.ReturnToSupplier, Common.LogonUser.LogonUserName, Common.UpdateFunction.UnitTests);
			
			tmpOSC.AddUnitToInvoice(BOL.OutgoingShipmentCollection.LineItem.GetLineItem(tmpBU.BloodUnitGuid));
			tmpOSC.SaveChangesToInvoiceItems(new System.Collections.ArrayList(), Common.UpdateFunction.UnitTests);

			Assert.IsTrue(_patientUnitSelection.IsOnPendingOutgoingShipmentList(tmpBU));
		}


		
		[Test]
		public void ValidateAntibodyAntigenRequirementsMet_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002" ,"E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);

			// First, delete all data currently out there for patient
			this.DeleteConflictingTrData(_orderedComponent.Patient, bloodUnit, false, true, false);

			BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.Technologist);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);

			// 1) No TRs for patient
			_patientUnitSelection.ValidateAntibodyAntigenRequirementsMet(bloodUnit);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// 2) TRs and unit has not been tested
			//this.SetUpData((Guid) dtRow["PatientGuid"], componentClass, true, true, "R", "R", true);
			this.SetUpData(_patientUnitSelection.OrderedComponent.Patient.PatientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true);
				
			this.SetupAntigenTr(_orderedComponent.Patient, bloodUnit.BloodUnitGuid);
			_patientUnitSelection.ValidateAntibodyAntigenRequirementsMet(bloodUnit);
			BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSM.OverRideable == true);
			_patientUnitSelection.ValidationMessages.Clear();

			// 3) TRs and unit has tested negative for TR
			this.SetupBloodUnitTests(bloodUnit.BloodUnitGuid, 212, "N");
			_patientUnitSelection.ValidateAntibodyAntigenRequirementsMet(bloodUnit);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Must recreate PUS with user that has priveleges to issue antigen Pos unit
			vbecsUser = this.GetVbecsUser(Common.UserRoleType.LeadTechnologist);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);

			// 4) TRs and unit has tested positive, but can override
			this.DeleteConflictingTrData(_orderedComponent.Patient, bloodUnit, false, false, true);
			this.SetupBloodUnitTests(bloodUnit.BloodUnitGuid, 212, "P");
			_patientUnitSelection.ValidateAntibodyAntigenRequirementsMet(bloodUnit);
			pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSM.OverRideable);
			_patientUnitSelection.ValidationMessages.Clear();

			// Done, so clean up data
			this.DeleteConflictingTrData(_orderedComponent.Patient, bloodUnit, true, true, true);

			// Wait, one more for new antibody message
			this.SetupAntibodyTr(_orderedComponent.Patient, bloodUnit.BloodUnitGuid);
			this.SetupBloodUnitTests(bloodUnit.BloodUnitGuid, 212, "P");

			// CR2899: recreate the object to account for caching
			_orderedComponent.Patient = new BOL.Patient(_orderedComponent.Patient.PatientGuid);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);

			_patientUnitSelection.ValidateAntibodyAntigenRequirementsMet(bloodUnit);
			pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSM.OverRideable == true);
			_patientUnitSelection.ValidationMessages.Clear();

			// Done, so clean up data
			this.DeleteConflictingTrData(_orderedComponent.Patient, bloodUnit, true, true, true);

		}
		[Test]
		public void ValidateAntibodyAntigenRequirementsMet_Fail()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002" ,"E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);

			// First, delete all data currently out there for patient
			this.DeleteConflictingTrData(_orderedComponent.Patient, bloodUnit, false, true, false);

			BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.EnhancedTechnologist);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);

			this.SetupAntigenTr(_orderedComponent.Patient, bloodUnit.BloodUnitGuid);
			this.SetupBloodUnitTests(bloodUnit.BloodUnitGuid, 212, "P");
			_patientUnitSelection.ValidateAntibodyAntigenRequirementsMet(bloodUnit);
			BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSM.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// clear out data
			this.DeleteConflictingTrData(_orderedComponent.Patient, bloodUnit, true, true, true);

			// Add antibdy
			this.SetupAntibodyTr(_orderedComponent.Patient, bloodUnit.BloodUnitGuid);
			this.SetupBloodUnitTests(bloodUnit.BloodUnitGuid, 212, "P");

			// CR2899: recreate the object to account for caching
			_orderedComponent.Patient = new BOL.Patient(_orderedComponent.Patient.PatientGuid);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);

			_patientUnitSelection.ValidateAntibodyAntigenRequirementsMet(bloodUnit);
			//pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSM.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
		
		}


		#region Component requirements

        [Test]
        public void ValidateTT304TRsCMVNeg_Pass()
        {
            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E038", "E3730V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.Other);

            // Patient doesn't have a CMV Neg requirement and is level 1
            BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.Technologist);	// Level 1
            _patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);
            _patientUnitSelection.ValidateTT304TRsSCAndCMVNeg(bloodUnit, false, false);		// Patient has no CMV requirement
            Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);


            // Patient has a CMV Neg requirement that isn't met, but at a high enough security for it to pass
            vbecsUser = this.GetVbecsUser(Common.UserRoleType.LeadTechnologist);	// Level 6
            _patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);
            _patientUnitSelection.ValidateTT304TRsSCAndCMVNeg(bloodUnit, true, false);
            BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage)_patientUnitSelection.ValidationMessages[0];
            Assert.IsTrue(pUSM.OverRideable == true &&
                pUSM.MessageText == "This unit does not currently satisfy the patient's CMV Negative Transfusion Requirement. Further processing is needed.");
        }
        [Test]
        public void ValidateTT304TRsCMVNeg_Fail()
        {
            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E3845V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.Other);

            // Patient has a CMV Neg requirement that isn't met and is low level security, so won't pass
            BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.Technologist);
            _patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);
            _patientUnitSelection.ValidateTT304TRsSCAndCMVNeg(bloodUnit, true, false);
            BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage)_patientUnitSelection.ValidationMessages[0];
            Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 &&
                pUSM.OverRideable == false &&
                pUSM.MessageText == "Patient requires CMV Negative blood. Selected unit is not CMV Negative. You do not have proper security to select this unit.");
        }


        
        
        [Test]
        public void ValidateTT304TRsCMVandSCNeg_Pass()
        {
            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);

            BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.LeadTechnologist);			// Level 6
            _patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);
            _patientUnitSelection.ValidateTT304TRsSCAndCMVNeg(bloodUnit, true, true);
            BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage)_patientUnitSelection.ValidationMessages[0];
            Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 &&
                pUSM.OverRideable == true &&
                pUSM.MessageText == "This unit does not currently satisfy the patient's CMV Negative or Sickle Cell Transfusion Requirements. Further processing is needed.");
        }
        [Test]
        public void ValidateTT304TRsCMVandSCNeg_Fail()
        {
            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
            _patientUnitSelection.ValidateTT304TRsSCAndCMVNeg(bloodUnit, true, true);
            BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage)_patientUnitSelection.ValidationMessages[0];
            Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 &&
                pUSM.OverRideable == false &&
                pUSM.MessageText == "Patient requires Sickle Cell and CMV Negative blood. Selected unit is not Sickle Cell or CMV Negative. You do not have proper security to select this unit.");
        }

        [Test]
        public void ValidateTT304TRsCMVandSCNeg_SicklePresent__Level3_Pass()
        {
            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
            DataCreator.CreateBloodUnitTest(Guid.Empty, bloodUnit.BloodUnitGuid, (int) Common.TestType.SickleCell, "N");

            BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.LeadTechnologist);			// Level 3
            _patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);
            _patientUnitSelection.ValidateTT304TRsSCAndCMVNeg(bloodUnit, true, true);
            BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage)_patientUnitSelection.ValidationMessages[0];
            Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 &&
                pUSM.OverRideable == true &&
                pUSM.MessageText == "This unit does not currently satisfy the patient's CMV Negative Transfusion Requirement. Further processing is needed.");
        }

        [Test]
        public void ValidateTT304TRsCMVandSCNeg_SicklePresent__Level1_Pass()
        {
            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
            DataCreator.CreateBloodUnitTest(Guid.Empty, bloodUnit.BloodUnitGuid, (int)Common.TestType.SickleCell, "N");

            BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.Technologist);			// Level 1
            _patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);
            _patientUnitSelection.ValidateTT304TRsSCAndCMVNeg(bloodUnit, true, true);
            BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage)_patientUnitSelection.ValidationMessages[0];
            Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 &&
                pUSM.OverRideable == false &&
                pUSM.MessageText == "Patient requires CMV Negative blood. Selected unit is not CMV Negative. You do not have proper security to select this unit.");
        }

        [Test]
        public void ValidateTT304TRsCMVandSCNeg_CMVPresent__Level3_Pass()
        {
            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
            DataCreator.CreateBloodUnitTest(Guid.Empty, bloodUnit.BloodUnitGuid, (int)Common.TestType.CMV, "N");

            BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.LeadTechnologist);			// Level 3
            _patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);
            _patientUnitSelection.ValidateTT304TRsSCAndCMVNeg(bloodUnit, true, true);
            BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage)_patientUnitSelection.ValidationMessages[0];
            Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 &&
                pUSM.OverRideable == true &&
                pUSM.MessageText == "This unit does not currently satisfy the patient's Sickle Cell Transfusion Requirement. Further processing is needed.");
        }

        [Test]
        public void ValidateTT304TRsCMVandSCNeg_CMVPresent__Level1_Pass()
        {
            BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
            DataCreator.CreateBloodUnitTest(Guid.Empty, bloodUnit.BloodUnitGuid, (int)Common.TestType.CMV, "N");

            BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.Technologist);			// Level 1
            _patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);
            _patientUnitSelection.ValidateTT304TRsSCAndCMVNeg(bloodUnit, true, true);
            BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage)_patientUnitSelection.ValidationMessages[0];
            Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 &&
                pUSM.OverRideable == false &&
                pUSM.MessageText == "Patient requires Sickle Cell Negative blood. Selected unit is not Sickle Cell Negative. You do not have proper security to select this unit.");
        }


		[Test]
		public void  ValidateTT304TRsLeukoReduced_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E028" ,"E3573V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.CRYO);
			_patientUnitSelection.ValidateTT304TRsLeukoReduced(bloodUnit);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
		}
		[Test]
		public void  ValidateTT304TRsLeukoReduced_Fail()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002" ,"E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			_patientUnitSelection.ValidateTT304TRsLeukoReduced(bloodUnit);
			BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSM.OverRideable);
		}


		[Test]
		public void  ValidateTT304TRsWashedPLT_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E027" ,"E4440V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.PLT);
			_patientUnitSelection.ValidateTT304TRsWashedPLT(bloodUnit);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
		}
		[Test]
		public void  ValidateTT304TRsWashedPLT_Fail()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E026" ,"E3465V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.PLT);
			_patientUnitSelection.ValidateTT304TRsWashedPLT(bloodUnit);
			BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSM.OverRideable);
		}


		[Test]
		public void  ValidateTT304TRsWashedRBC_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E003" ,"E0474V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			_patientUnitSelection.ValidateTT304TRsWashedRBC(bloodUnit);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
		}
		[Test]
		public void  ValidateTT304TRsWashedRBC_Fail()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002" ,"E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			_patientUnitSelection.ValidateTT304TRsWashedRBC(bloodUnit);
			BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSM.OverRideable);
		}


		[Test]
		public void  ValidateTT304TRsIrradiateCells_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E003" ,"E0306V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			_patientUnitSelection.ValidateTT304TRsIrradiateCells(bloodUnit);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
		}
		[Test]
		public void  ValidateTT304TRsIrradiateCells_Fail()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002" ,"E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			_patientUnitSelection.ValidateTT304TRsIrradiateCells(bloodUnit);
			BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSM.OverRideable);
		}

		#endregion


		#region validate whole blood (TT_301)

		[Test]
		public void ValidateWholeBloodTT301_Pass()
		{
			
			BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.LeadTechnologist);
			vbecsUser.LogonToDivision( _orderedComponent.DivisionCode );
			vbecsUser.CurrentDivisionRole.Division.ServiceTypeIndicator = true;

			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);

			#region patient is O Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			BOL.PatientUnitSelectionMessage pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is O Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion
		}

		[Test]
		public void ValidateWholeBloodTT301_Fail()
		{
			BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.EnhancedTechnologist);
			vbecsUser.LogonToDivision( _orderedComponent.DivisionCode );
			vbecsUser.CurrentDivisionRole.Division.ServiceTypeIndicator = true;

			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);

			#region patient is O Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			BOL.PatientUnitSelectionMessage pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is ???

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.NA;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.NotProvided;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateWholeBloodTT301(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion
		}

		#endregion


		#region validate RBC (TT_302)

		[Test]
		public void ValidateNonWholeBloodNonPlasmaTT302_Pass()
		{
			BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.LeadTechnologist);
			vbecsUser.LogonToDivision( _orderedComponent.DivisionCode );
			vbecsUser.CurrentDivisionRole.Division.ServiceTypeIndicator = true;

			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);


			#region patient is O Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			BOL.PatientUnitSelectionMessage pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is O Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is I Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is I Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is O Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is I Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is none of the above

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.NA;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.NotProvided;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion
		}
		[Test]
		public void ValidateNonWholeBloodNonPlasmaTT302_Fail()
		{
			BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.EnhancedTechnologist);
			vbecsUser.LogonToDivision( _orderedComponent.DivisionCode );
			vbecsUser.CurrentDivisionRole.Division.ServiceTypeIndicator = true;

			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);


			#region patient is O Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			BOL.PatientUnitSelectionMessage pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is O Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is I Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is I Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is O Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is I Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is none of the above

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.NA;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.NotProvided;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is I Positive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is I Negative

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is O Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is I Inconclusive

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Inconclusive;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is none of the above

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.NA;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.NotProvided;

			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// All other types are allowed only with override
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidateNonWholeBloodNonPlasmaTT302(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion
		}

		#endregion



		#region validate plasma (TT_302A)

		// CR3097: Added code to account for pooled types (Mx)
		[Test]
		public void ValidatePlasmaOnlyTT303A_Pass()
		{
			BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.LeadTechnologist);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);

			#region patient is O Pos
			// A, B, AB, O and Pooled allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();			
			
			#endregion



			#region patient is O Neg
			// A, B, AB, O and Pooled allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion



			#region patient is A Pos
			// A, AB and Pooled allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			#endregion



			#region patient is A Neg

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion



			#region patient is B Pos
			// B, AB and Pooled allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion



			#region patient is B Neg
			// B, AB and Pooled allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			
			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			#endregion



			#region patient is AB Pos
			// AB and Pooled allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			

			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			#endregion



			#region patient is AB Neg
			// AB and Pooled allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

		
			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion



			#region patient is I Pos
			// AB and Pooled allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion



			#region patient is I Pos
			// AB and Pooled allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
		
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion



			#region patient is NA NA
			// AB and Pooled allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.NA;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.NotProvided;
			
			
			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion
		}


		[Test]
		public void ValidatePlasmaOnlyTT303A_Fail()
		{
			BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.EnhancedTechnologist);

			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);

			#region patient is O Pos
			// I and NA not allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			BOL.PatientUnitSelectionMessage pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion



			#region patient is O Neg
			// I and NA not allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion


			#region patient is A Pos
			// B, O, I and NA not allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion


			#region patient is A Neg
			// B, O, I and NA not allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			

			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
		
		
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion


			#region patient is B Pos
			// A, O, I and NA not allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion


			#region patient is B Neg
			// A, O, I and NA not allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion


			#region patient is AB Pos
			// A, B, O, I and NA not allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

		
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion


			#region patient is AB Neg
			// A, B, O, I and NA not allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion


			#region patient is I Pos
			// A, B, O, I and NA not allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();
		

			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion


			#region patient is I Pos
			// A, B, O, I and NA not allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			

			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is NA NA
			// A, B, O, I and NA not allowed

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.NA;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.NotProvided;
			
			
			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);
			
			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();


			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303A(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion
		}

		#endregion


		#region validate plasma (TT_302B)

		[Test]
		public void ValidatePlasmaOnlyTT303B_Pass()
		{
			BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.LeadTechnologist);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);


			#region patient is O Pos

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			#endregion

			#region patient is O Neg

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Pos

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Neg

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion		

			#region patient is B Pos

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Neg

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion		

			#region patient is AB Pos

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Neg

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion		

			#region patient is I Pos

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is I Neg

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			// O Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// O Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// A Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// A Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// B Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// B Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// AB Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// AB Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.AB;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();


			// Pooled Pos (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			// Pooled Neg (CR3097)
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.Mx;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion
		}

		[Test]
		public void ValidatePlasmaOnlyTT303B_Fail()
		{
			BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.Technologist);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);


			#region patient is O Pos

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			BOL.PatientUnitSelectionMessage pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is O Neg

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Pos

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is A Neg

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.A;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Pos

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is B Neg

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.B;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Pos

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is AB Neg

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.AB;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is I Pos

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is I Neg

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.I;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is NA Pos

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.NA;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion

			#region patient is NA Neg

			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.NA;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			
			// I Pos
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// I Neg
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.I;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			// NA NA
			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.NA;
			_genericBU.BloodUnitMedia.AboRh.RH = Common.RH.NotProvided;
			_patientUnitSelection.ValidatePlasmaOnlyTT303B(_genericBU);

			pUSMessage = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSMessage.OverRideable == false);
			_patientUnitSelection.ValidationMessages.Clear();

			#endregion
		}

		#endregion


		[Test]
		public void ValidateUnitRhRequirement_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Negative, Common.ComponentClass.RBC);
			_patientUnitSelection.ValidateUnitRhRequirement(bloodUnit);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
		}
		[Test]
		public void ValidateUnitRhRequirement_Fail()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			_patientUnitSelection.ValidateUnitRhRequirement(bloodUnit);
			BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSM.OverRideable == true);
		}


		[Test]
		public void IsFrozenUnit_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			_patientUnitSelection.IsFrozenUnit(bloodUnit);
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
		}
		[Test]
		public void IsFrozenUnit_Fail()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E004", "E4199V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			_patientUnitSelection.IsFrozenUnit(bloodUnit);
			BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1 && pUSM.OverRideable);
		}


		[Test]
		public void QuarantinedUnit_Pass()
		{
			_genericBU.BloodUnitStatus.QuarantineIndicator = false;
			Assert.IsTrue(_patientUnitSelection.QuarantinedUnit(_genericBU) == false && _patientUnitSelection.ValidationMessages.Count == 0);
		}
		[Test]
		public void QuarantinedUnit_Fail()
		{
			_genericBU.BloodUnitStatus.QuarantineIndicator = true;
			Assert.IsTrue(_patientUnitSelection.QuarantinedUnit(_genericBU) == true && 
				((BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0]).OverRideable == false);
		}


		[Test]
		public void ExpiredUnit_Pass()
		{
			BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.EnhancedTechnologist);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);

			_genericBU.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddDays(2);
			Assert.IsTrue(_patientUnitSelection.ExpiredUnit(_genericBU, DateTime.Now) == false);

			_genericBU.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddDays(-2);
			Assert.IsTrue(_patientUnitSelection.ExpiredUnit(_genericBU, DateTime.Now) == true );
			BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(pUSM.OverRideable == true);
		}
		[Test]
		public void ExpiredUnit_Fail()
		{
			BOL.VbecsUser vbecsUser = this.GetVbecsUser(Common.UserRoleType.Technologist);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, vbecsUser);

			_genericBU.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddDays(-2);
			Assert.IsTrue(_patientUnitSelection.ExpiredUnit(_genericBU, DateTime.Now) == true);
			BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(pUSM.OverRideable == false);
		}

		/*
		[Test]
		public void ValidateCurrentSpecimenWithOwnResults_Pass()
		{
			// This has already been thoroughly covered by the corresponding specimen test case, so 
			// in depth testing isn't required here. 
			_patientUnitSelection.OrderedComponent.Specimen = new BOL.Specimen();
			Common.TypingTestResults result = _patientUnitSelection.ValidateCurrentSpecimenWithOwnResults();
			Assertion.IsTrue(result == Common.TypingTestResults.NoResults);
		}
		[Test]
		public void ValidateCurrentSpecimenWithOwnResults_Fail()
		{
			// This has already been thoroughly covered by the corresponding specimen test case, so 
			// in depth testing isn't required here. 
			_patientUnitSelection.OrderedComponent.Specimen = null;
			Common.TypingTestResults result = _patientUnitSelection.ValidateCurrentSpecimenWithOwnResults();
			Assertion.IsTrue(result == Common.TypingTestResults.NoResults);
		}
		


		[Test]
		public void ValidateCurrentSpecimenWithHistoricResults_Pass()
		{
			// This has already been thoroughly covered by the corresponding specimen test case, so 
			// in depth testing isn't required here. 
			_patientUnitSelection.OrderedComponent.Specimen = new BOL.Specimen();
			Common.TypingTestResults result = _patientUnitSelection.ValidateCurrentSpecimenWithHistoricResults();
			Assertion.IsTrue(result == Common.TypingTestResults.NoResults);
		}
		[Test]
		public void ValidateCurrentSpecimenWithHistoricResults_Fail()
		{
			// This has already been thoroughly covered by the corresponding specimen test case, so 
			// in depth testing isn't required here. 
			_patientUnitSelection.OrderedComponent.Specimen = null;
			Common.TypingTestResults result = _patientUnitSelection.ValidateCurrentSpecimenWithHistoricResults();
			Assertion.IsTrue(result == Common.TypingTestResults.Match);
		}
		*/


		[Test]
		public void GetTimeUntilExpiration_Pass()
		{
			_genericBU.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddDays(3);
			string time = _patientUnitSelection.GetTimeUntilExpiration(_genericBU);
			Assert.IsNotNull(time != null && time.Length > 0);

			_genericBU.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddDays(1.5);
			time = _patientUnitSelection.GetTimeUntilExpiration(_genericBU);
			Assert.IsNotNull(time != null && time.Length > 0);

			_genericBU.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddHours(3);
			time = _patientUnitSelection.GetTimeUntilExpiration(_genericBU);
			Assert.IsNotNull(time != null && time.Length > 0);

			_genericBU.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddHours(1.5);
			time = _patientUnitSelection.GetTimeUntilExpiration(_genericBU);
			Assert.IsNotNull(time != null && time.Length > 0);

			_genericBU.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddMinutes(2);
			time = _patientUnitSelection.GetTimeUntilExpiration(_genericBU);
			Assert.IsNotNull(time != null && time.Length > 0);

			_genericBU.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddDays(-2);
			time = _patientUnitSelection.GetTimeUntilExpiration(_genericBU);
			Assert.IsNotNull(time != null && time.Length > 0 && time == "Expired");
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void GetTimeUntilExpiration_Fail()
		{
			_patientUnitSelection.GetTimeUntilExpiration(null);
		}

		[Test]
		public void AddOptionalCrossmatch_Pass()
		{
			_patientUnitSelection.SetInitialBrokenRules("FrmSelectUnitsForPatient");

			_orderedComponent.AssociateWithSpecimen(null);
			int brokenRulesBefore =_patientUnitSelection.BrokenRulesCount;

			_patientUnitSelection.AddOptionalCrossmatch(System.Guid.NewGuid());
			Assert.IsTrue(brokenRulesBefore+1 == _patientUnitSelection.BrokenRulesCount);

			// reset and try again with non-null specimen
			_orderedComponent.AssociateWithSpecimen(new BOL.Specimen());
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));
			_patientUnitSelection.AddOptionalCrossmatch(System.Guid.NewGuid());
			Assert.IsTrue(brokenRulesBefore == _patientUnitSelection.BrokenRulesCount);	
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void AddOptionalCrossmatch_Fail()
		{
			_patientUnitSelection.AddOptionalCrossmatch(System.Guid.Empty);
		}


		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void LoadDataRowFromThis_Pass()
		{
			_patientUnitSelection.LoadDataRowFromThis(new DataTable().NewRow());
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{
		}


		[Test]
		public void RemoveOptionalCrossmatch_Pass()
		{
			_patientUnitSelection.SetInitialBrokenRules("FrmSelectUnitsForPatient");
			_orderedComponent.AssociateWithSpecimen(null);

			Guid unitGuid = Guid.NewGuid();
			_patientUnitSelection.AddOptionalCrossmatch(unitGuid);
			_patientUnitSelection.RemoveOptionalCrossmatch(unitGuid);
			
			// Second condition - specimen associated with Ordered Component
			_orderedComponent.AssociateWithSpecimen(new BOL.Specimen());

			unitGuid = Guid.NewGuid();
			_patientUnitSelection.AddOptionalCrossmatch(unitGuid);
			_patientUnitSelection.RemoveOptionalCrossmatch(unitGuid);

			Assert.IsTrue(true, "ReturnValue");
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void RemoveOptionalCrossmatch_Fail()
		{
			_patientUnitSelection.RemoveOptionalCrossmatch(Guid.Empty);
		}

		[Test]
		public void OptionalCrossmatchOrdered_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			_patientUnitSelection.AddOptionalCrossmatch(unitGuid);
			Assert.IsTrue(_patientUnitSelection.OptionalCrossmatchOrdered(unitGuid));
		}

		[Test]
		public void OptionalCrossmatchOrdered_Fail()
		{
			Guid unitGuid = Guid.NewGuid();
			Assert.IsTrue(!_patientUnitSelection.OptionalCrossmatchOrdered(unitGuid));
		}

		[Test]
		public void OptionalCrossmatchOrdered_Guid_Pass()
		{
			bool tstValue = _patientUnitSelection.OptionalCrossmatchOrdered(_genericBU.BloodUnitGuid);
			Assert.IsNotNull(tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void OptionalCrossmatchOrdered_Guid_Fail()
		{
		}


		[Test]
		public void ValidSpecimenExists_Pass()
		{
			_orderedComponent.GetSpecimen().UnacceptableSpecimenIndicator = false;
			_patientUnitSelection.OrderedComponent.ComponentOrdered.BloodComponentClassId = (int) Common.ComponentClass.RBC;
			_orderedComponent.ComponentOrdered.IsSpecimenRequired = true;

			Assert.IsTrue( _patientUnitSelection.ValidSpecimenExists() );
		}
		[Test]
		public void ValidSpecimenExists_Fail()
		{
			// Unacceptable specimen
			_orderedComponent.GetSpecimen().UnacceptableSpecimenIndicator = true;
			_patientUnitSelection.OrderedComponent.ComponentOrdered.BloodComponentClassId = (int) Common.ComponentClass.RBC;
			_orderedComponent.ComponentOrdered.IsSpecimenRequired = true;
			Assert.IsTrue(!_patientUnitSelection.ValidSpecimenExists() );



			// New specimen
			// reset
			_orderedComponent.GetSpecimen().UnacceptableSpecimenIndicator = false;
			
			_orderedComponent.GetSpecimen().IsNew = true;
			Assert.IsTrue(!_patientUnitSelection.ValidSpecimenExists() );


			
			// Unacceptable specimen
			//reset
			_orderedComponent.GetSpecimen().IsNew = false;

			_orderedComponent.GetSpecimen().UnacceptableSpecimenIndicator = true;
			Assert.IsTrue(!_patientUnitSelection.ValidSpecimenExists() );



			// Not maintainable
			//reset
			_orderedComponent.GetSpecimen().UnacceptableSpecimenIndicator = false;

			_orderedComponent.GetSpecimen().MaintainableSpecimenIndicator = false;
			Assert.IsTrue(!_patientUnitSelection.ValidSpecimenExists() );



			// Null specimen
			//reset
			_orderedComponent.GetSpecimen().MaintainableSpecimenIndicator = true;

			_orderedComponent.AssociateWithSpecimen(null);
			Assert.IsTrue(!_patientUnitSelection.ValidSpecimenExists() );
		}

		#region electronic crossmatch

		#region DivisionAndPatient eXM (determines eXM status for the division and patient)

		// BR_3.17, BR_3.18
		[Test]
		public void GetDivisionAndPatientEXMStatus_Pass()
		{
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);

			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMEnabled().ResString) && message.IndexOf("exception") == -1);
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_Fail_Discrepancy()
		{	
			// This test ensures code coverage of 100%
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			
			// Insert tests
			Guid orderedTestGuid = Guid.NewGuid();
			Guid orderedTestGuidTwo = Guid.NewGuid();
			Guid specimenGuid = this.SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "18", "AB");
			Guid specimenGuidTwo = this.SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "21", "N");

			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid IN ('" + specimenGuid + "','" + specimenGuidTwo + "')");
	
			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledABORhDiscrepancy().ResString) && message.IndexOf("exception") == -1);
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_Fail_Discrepancy_Two()
		{	
			// This test ensures code coverage of 100%
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			
			// Insert tests
			Guid orderedTestGuid = Guid.NewGuid();
			Guid orderedTestGuidTwo = Guid.NewGuid();
			Guid specimenGuid = this.SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "18", "A");
			Guid specimenGuidTwo = this.SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "21", "N");

			UnitTestUtil.RunSQL("UPDATE SpecimenTest SET AboRhChangeIndicator = 1 WHERE SpecimenTestGuid = '" + specimenGuid + "'");

			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid IN ('" + specimenGuid + "','" + specimenGuidTwo + "')");
	
			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledABORhDiscrepancy().ResString) && message.IndexOf("exception") == -1);
		}
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void GetDivisionAndPatientEXMStatus_DivisionNBull_Fail()
		{
			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(null);
			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.UnitEXMDisabled().ResString) && message.IndexOf("exception") == -1);			
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_DivisionNotEnabled_Fail()
		{
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = false;
			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);
			Assert.IsTrue(message == Common.StrRes.InfoMsg.Common.UnitEXMDisabled().ResString);			
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_NoSpecimen_Fail()
		{
			
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			_patientUnitSelection.OrderedComponent.GetSpecimen().IsNew = true;
			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);

			Assert.IsTrue(message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledNoSpecimen().ResString);			
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_NoSpecimenResults_Fail()
		{
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE PatientSpecimenGuid = '" + _orderedComponent.GetSpecimen().SpecimenGuid + "'");
			
			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);			

			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledNoSpecimenTesting().ResString) && message.IndexOf("exception") == -1 );
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_AntibodyIndentified_Fail()
		{
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			//UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE PatientSpecimenGuid = '" + _orderedComponent.Specimen.SpecimenGuid + "'");
			
			Guid specimenGuid = this.SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "52", null);
			Guid specimenGuidTwo = this.SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "52", "P");

			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);
			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledPositiveAntibody().ResString) && message.IndexOf("exception") == -1 );
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid IN ('" + specimenGuid + "','" + specimenGuidTwo + "')");
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_TransfusionRequirement_AntibodyHistory_Fail()
		{	
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			this.SetAntiD(_orderedComponent.Patient, Guid.Empty, false);

			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);
			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledAntibodyHistory().ResString) && message.IndexOf("exception") == -1);
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_TransfusionRequirementFromConversion_AntibodyHistory_Fail()
		{	
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			this.SetAntiD(_orderedComponent.Patient, Guid.Empty, true);

			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);
			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledAntibodyHistory().ResString) && message.IndexOf("exception") == -1);
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_TransfusionRequirement_AntigenNegative_Fail()
		{	
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			this.SetupAntigenTr(_orderedComponent.Patient, Guid.Empty);

			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);
			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledAntigenNegative().ResString) && message.IndexOf("exception") == -1);
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_ABORhHistory_TypingDifficulty_ABO_Fail()
		{	
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			Guid specimenTestGuid = this.SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "18", "I");
			Guid specimenTestGuidTwo = this.SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "18", null);

			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);
			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledTypingDifficulty().ResString) && message.IndexOf("exception") == -1);
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid IN ('" + specimenTestGuid  + "','" + specimenTestGuidTwo + "')");
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_ABORhHistory_TypingDifficulty_Rh_Fail()
		{	
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			Guid specimenTestGuid = this.SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "21", "I");
			Guid specimenTestGuidTwo = this.SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "21", null);

			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);
			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledTypingDifficulty().ResString) && message.IndexOf("exception") == -1);
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid IN ('" + specimenTestGuid + "','" + specimenTestGuidTwo + "')");
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_ABORhHistory_Discrepancy_ABO_Fail()
		{	
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			//UnitTestUtil.RunSQL("DELETE * FROM SpecimenTest WHERE PatientSpecimenGuid = '" + _orderedComponent.Specimen.SpecimenGuid+ "'");
			Guid specimenTestGuid = this.SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "18", "B");
	
			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);
			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledABORhDiscrepancy().ResString) && message.IndexOf("exception") == -1);
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + specimenTestGuid + "'");
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_ABORhHistory_Discrepancy_Rh_Fail()
		{	
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			//UnitTestUtil.RunSQL("DELETE * FROM SpecimenTest WHERE PatientSpecimenGuid = '" + _orderedComponent.Specimen.SpecimenGuid+ "'");
			Guid specimenTestGuid = this.SetupSpecimenTest(_orderedComponent.GetSpecimen().SpecimenGuid, _orderedTestGuid, "21", "N");
	
			string message = _patientUnitSelection.GetDivisionAndPatientEXMStatus(division);
			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledABORhDiscrepancy().ResString) && message.IndexOf("exception") == -1);
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + specimenTestGuid + "'");
		}
		[Test]
		public void GetDivisionAndPatientEXMStatus_ABORhHistory_NotEnoughTypings_Fail()
		{	
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			division.ElectronicCrossmatchIndicator = true;
			UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE PatientSpecimenGuid = '" + _specimenOneGuid + "'");
	
			string message = this._patientUnitSelection.GetDivisionAndPatientEXMStatus(division);
			Assert.IsTrue( (message == Common.StrRes.InfoMsg.Common.PatienteXMDisabledNeedTwoTypings().ResString) && message.IndexOf("exception") == -1);
		}


		#endregion

		#region Unit eXM (determines eXM status for the unit)

		[Test]
		public void GetUnitEXMStatus_Pass()
		{
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			bool status = _patientUnitSelection.GetUnitEXMStatus(bloodUnit);
			Assert.IsTrue(status==true);

			// Now, try with WB
			_orderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E001", "E0001V00", Common.ABO.A, Common.RH.Negative, Common.ComponentClass.WB);
			status = _patientUnitSelection.GetUnitEXMStatus(bloodUnit);
			Assert.IsTrue(status==true);			
		}
		[Test]
		public void GetUnitEXMStatus_Fail()
		{
			// Incompatible RBC (ABO)
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.B, Common.RH.Positive, Common.ComponentClass.RBC);
			
			string sql = "INSERT INTO BloodUnitTest (TestDate, BloodUnitTestGuid, BloodUnitGuid, BloodTestTypeId, TestResultId, EntryTechID, EntryMethodCode, RecordStatusCode, DivisionCode, LastUpdateDate, LastUpdateUser) " + 
				"VALUES (getutcdate(), '" + Guid.NewGuid() + "','" + bloodUnit.BloodUnitGuid + "'," + 18 + ",'A','" + "Foo" + "','M','A','" + UnitTestConfig.TestDivisionCode + "','" + DateTime.Now.ToString() + "','UnitTest')";
			UnitTestUtil.RunSQL(sql);

			bool status = _patientUnitSelection.GetUnitEXMStatus(bloodUnit);
			BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage) _patientUnitSelection.ValidationMessages[0];
			Assert.IsTrue(!status);


			// Incompatible RBC (Rh)
			_orderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Negative, Common.ComponentClass.RBC);
			
			sql = "INSERT INTO BloodUnitTest (TestDate, BloodUnitTestGuid, BloodUnitGuid, BloodTestTypeId, TestResultId, EntryTechID, EntryMethodCode, RecordStatusCode, DivisionCode, LastUpdateDate, LastUpdateUser) " + 
				"VALUES (getutcdate(), '" + Guid.NewGuid() + "','" + bloodUnit.BloodUnitGuid + "'," + 21 + ",'P','" + "Foo" + "','M','A','" + UnitTestConfig.TestDivisionCode + "','" + DateTime.Now.ToString() + "','UnitTest')";
			UnitTestUtil.RunSQL(sql);

			status = _patientUnitSelection.GetUnitEXMStatus(bloodUnit);
			Assert.IsTrue(!status);


			// Incompatible WB (ABO)
			bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E001", "E0001V00", Common.ABO.B, Common.RH.Positive, Common.ComponentClass.WB);
			
			sql = "INSERT INTO BloodUnitTest (TestDate, BloodUnitTestGuid, BloodUnitGuid, BloodTestTypeId, TestResultId, EntryTechID, EntryMethodCode, RecordStatusCode, DivisionCode, LastUpdateDate, LastUpdateUser) " + 
				"VALUES (getutcdate(), '" + Guid.NewGuid() + "','" + bloodUnit.BloodUnitGuid + "'," + 18 + ",'A','" + "Foo" + "','M','A','" + UnitTestConfig.TestDivisionCode + "','" + DateTime.Now.ToString() + "','UnitTest')";
			UnitTestUtil.RunSQL(sql);
			
			status = _patientUnitSelection.GetUnitEXMStatus(bloodUnit);
			Assert.IsTrue(!status);


			// Incompatible WB (Rh)
			_orderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E001", "E0001V00", Common.ABO.A, Common.RH.Negative, Common.ComponentClass.WB);
			
			sql = "INSERT INTO BloodUnitTest (TestDate, BloodUnitTestGuid, BloodUnitGuid, BloodTestTypeId, TestResultId, EntryTechID, EntryMethodCode, RecordStatusCode, DivisionCode, LastUpdateDate, LastUpdateUser) " + 
				"VALUES (getutcdate(), '" + Guid.NewGuid() + "','" + bloodUnit.BloodUnitGuid + "'," + 21 + ",'P','" + "Foo" + "','M','A','" + UnitTestConfig.TestDivisionCode + "','" + DateTime.Now.ToString() + "','UnitTest')";
			UnitTestUtil.RunSQL(sql);
			
			status = _patientUnitSelection.GetUnitEXMStatus(bloodUnit);
			Assert.IsTrue(!status);
		}

		#endregion

		#region Unit eXM text (set text that should be displayed for unit eXM)

		[Test]
		public void GetUnitEXMStatusText_Pass()
		{
			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Negative, Common.ComponentClass.RBC);
		
			string message = string.Empty;

			message = _patientUnitSelection.GetUnitEXMStatusText(bloodUnit);
			if ( new BOL.Division(_orderedComponent.DivisionCode).ServiceTypeIndicator == true)
			{
				Assert.IsTrue(message == Common.StrRes.InfoMsg.Common.UnitEXMEnabled().ResString && message.IndexOf("exception") == -1);
			}
			else
			{
				Assert.IsTrue(message == Common.StrRes.InfoMsg.Common.UnitEXMDisabled().ResString && message.IndexOf("exception") == -1);
			}
		}
		[Test]
		public void GetUnitEXMStatusText_Fail()
		{
			string message = string.Empty;

			BOL.Division division = new BOL.Division(_orderedComponent.DivisionCode);
			BOL.BloodUnit bloodUnitOne = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Negative, Common.ComponentClass.RBC);
			UnitTestUtil.RunSQL("DELETE FROM BloodUnitTest WHERE BloodUnitGuid = '" + bloodUnitOne.BloodUnitGuid + "'");

			message = _patientUnitSelection.GetUnitEXMStatusText(bloodUnitOne);
			Assert.IsTrue(message == Common.StrRes.InfoMsg.Common.UnitEXMDisabledNoABORhConfirm().ResString);

			BOL.BloodUnit bloodUnitTwo = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			//UnitTestUtil.RunSQL("UPDATE BloodUnitTest WHERE SET TestResultId = 'P' WHERE BloodUnitGuid = '" + bloodUnitTwo.BloodUnitGuid + "'");
			bloodUnitTwo.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			message = _patientUnitSelection.GetUnitEXMStatusText(bloodUnitTwo);
			Assert.IsTrue(message == Common.StrRes.InfoMsg.Common.UnitEXMDisabledRhConfirmNoMatch().ResString);

			bloodUnitTwo.BloodUnitMedia.AboRh.Abo = Common.ABO.O;
			message = _patientUnitSelection.GetUnitEXMStatusText(bloodUnitTwo);
			Assert.IsTrue(message == Common.StrRes.InfoMsg.Common.UnitEXMDisabledABOConfirmNoMatch().ResString);



			//			//_orderedComponent.Division.ServiceTypeIndicator = true; 
			//			string message = string.Empty;
			//
			//			//this.SetUpBloodUnitAndPatient();
			//			// by default, the patient and blood unit are both set up to be A Positive.  Set unit to B to generate error
			//			_genericBU.BloodUnitMedia.AboRh.Abo = Common.ABO.B;
			//			message = _patientUnitSelection.GetUnitEXMStatusText(_genericBU);
			//
			//			if ( new BOL.Division(_orderedComponent.DivisionCode).ServiceTypeIndicator == true)
			//			{
			//				Assertion.IsTrue(message == Common.StrRes.InfoMsg.Common.UnitEXMDisabledABOIncompatible().ResString && message.IndexOf("exception") == -1);
			//			}
		}

		#endregion


		[Test]
		public void GetCompatibleUnitsForDisplay_Pass()
		{
			Guid orderedComponentGuid = this.CreateTestData(_orderedComponent.PatientGuid, Common.ComponentClass.RBC, true, true, "N", "R", false);
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			Guid orderedUnitGuid = this.SetUpOrderedUnitData(bloodUnit.BloodUnitGuid, orderedComponentGuid, true, "C");

			// Set up OrderedComponent
			DataRow dtRow = UnitTestUtil.RunSQLGetDataRow("SELECT * FROM OrderedComponent WHERE OrderedComponentGuid = '" + orderedComponentGuid + "'");
			dtRow.Table.Columns.Add(TABLES.Patient.PatientGuid, typeof(System.Guid));
			dtRow[TABLES.Patient.PatientGuid] = _orderedComponent.PatientGuid;
			BOL.OrderedComponent oC = new BOL.OrderedComponent(dtRow);


			// RBC
			_orderedComponent.ComponentOrdered.BloodComponentClassId = (int) Common.ComponentClass.RBC;
			_patientUnitSelection = new gov.va.med.vbecs.BOL.PatientUnitSelection(oC, this.GetVbecsUser(Common.UserRoleType.Technologist));
			DataTable dt = _patientUnitSelection.GetCompatibleUnitsForDisplay();
			Assert.IsTrue(dt != null);

			UnitTestUtil.RunSQL("DELETE FROM OrderedUnit WHERE OrderedUnitGuid = '" + orderedUnitGuid+ "'");




			// FFP
			_orderedComponent.ComponentOrdered.BloodComponentClassId = (int) Common.ComponentClass.FFP;
			_patientUnitSelection = new gov.va.med.vbecs.BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));
			dt = _patientUnitSelection.GetCompatibleUnitsForDisplay();
			Assert.IsTrue(dt != null);

			// CRYO
			_orderedComponent.ComponentOrdered.BloodComponentClassId = (int) Common.ComponentClass.CRYO;
			_patientUnitSelection = new gov.va.med.vbecs.BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));
			dt = _patientUnitSelection.GetCompatibleUnitsForDisplay();
			Assert.IsTrue(dt != null);

			// Platelets
			_orderedComponent.ComponentOrdered.BloodComponentClassId = (int) Common.ComponentClass.PLT;
			_patientUnitSelection = new gov.va.med.vbecs.BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));
			dt = _patientUnitSelection.GetCompatibleUnitsForDisplay();
			Assert.IsTrue(dt != null);

			// Other
			_orderedComponent.ComponentOrdered.BloodComponentClassId = (int) Common.ComponentClass.Other;
			_patientUnitSelection = new gov.va.med.vbecs.BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));
			dt = _patientUnitSelection.GetCompatibleUnitsForDisplay();
			Assert.IsTrue(dt != null);

			// WB
			_orderedComponent.ComponentOrdered.BloodComponentClassId = (int) Common.ComponentClass.WB;
			_patientUnitSelection = new gov.va.med.vbecs.BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));
			dt = _patientUnitSelection.GetCompatibleUnitsForDisplay();
			Assert.IsTrue(dt != null);
		}
		[Test]
		public void GetCompatibleUnitsForDisplay_NoRestrictedUnits_Fail()
		{
			this.RefreshData = true;

			System.Guid patientGuid = this.CreatePatient();
			System.Guid directedPatientGuid = this.CreatePatient();

			UnitTestUtil.RunSQL(_cleanupSQL);
			this.SetUpData(patientGuid, Common.ComponentClass.RBC, true, true, "R", "R", true);
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			// Insert unit directed for another patient
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + directedPatientGuid.ToString() + "' WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'");
			
			// Test
			DataTable dtUnits = _patientUnitSelection.GetCompatibleUnitsForDisplay();
			Assert.IsTrue(dtUnits.Select(TABLES.BloodUnit.BloodUnitGuid + " = '" + bloodUnit.BloodUnitGuid + "'").Length == 0);
		}
		[Test]
		public void GetCompatibleUnitsForDisplay_EI_RBC_Fail()
		{
			// Unit should not appear because this is an EI situation (incompatible crossmatch and the
			// unit is RBC and not type O)
			this.RefreshData = true;

			System.Guid patientGuid = this.CreatePatient();

			UnitTestUtil.RunSQL(_cleanupSQL);
			this.SetUpData(patientGuid, Common.ComponentClass.RBC, true, false, "R", "R", true);
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			
			// Test
			DataTable dtUnits = _patientUnitSelection.GetCompatibleUnitsForDisplay();
			Assert.IsTrue(dtUnits.Select(TABLES.BloodUnit.BloodUnitGuid + " = '" + bloodUnit.BloodUnitGuid + "'").Length == 0);
		}
		[Test]
		public void GetCompatibleUnitsForDisplay_EI_FFP_Fail()
		{
			// Unit should not appear because this is an EI situation (incompatible crossmatch and the
			// unit is FFP and not type O)
			this.RefreshData = true;

			System.Guid patientGuid = this.CreatePatient();

			UnitTestUtil.RunSQL(_cleanupSQL);
			this.SetUpData(patientGuid, Common.ComponentClass.FFP, true, false, "R", "R", true);
			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E010", "E0749V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.FFP);
			
			// Test
			DataTable dtUnits = _patientUnitSelection.GetCompatibleUnitsForDisplay();
			Assert.IsTrue(dtUnits.Select(TABLES.BloodUnit.BloodUnitGuid + " = '" + bloodUnit.BloodUnitGuid + "'").Length == 0);
		}


		#region unit selection validation

		[Test]
		public void ValidateUnits_RBC_Pass()
		{
			this.ValidateUnitPass(Common.ComponentClass.RBC);
		}
		[Test]
		public void ValidateUnits_RBC_Fail()
		{
			this.ValidateUnitFail(Common.ComponentClass.RBC);
		}

		[Test]
		public void ValidateUnits_WB_Pass()
		{
			this.ValidateUnitPass(Common.ComponentClass.WB);
		}
		[Test]
		public void ValidateUnits_WB_Fail()
		{
			this.ValidateUnitFail(Common.ComponentClass.WB);	
		}

		[Test]
		public void ValidateUnits_FFP_Pass()
		{
			this.ValidateUnitPass(Common.ComponentClass.FFP);
		}
		[Test]
		public void ValidateUnits_FFP_Fail()
		{
			this.ValidateUnitFail(Common.ComponentClass.FFP);
		}

		[Test]
		public void ValidateUnits_CRYO_Pass()
		{
			this.ValidateUnitPass(Common.ComponentClass.CRYO);
		}
		[Test]
		public void ValidateUnits_CRYO_Fail()
		{
			this.ValidateUnitFail(Common.ComponentClass.CRYO);
		}
		
		[Test]
		public void ValidateUnits_PLT_Pass()
		{
			this.ValidateUnitPass(Common.ComponentClass.PLT);
		}
		[Test]
		public void ValidateUnits_PLT_Fail()
		{
			this.ValidateUnitFail(Common.ComponentClass.PLT);
		}
		
		[Test]
		public void ValidateUnits_Other_Pass()
		{
			this.ValidateUnitPass(Common.ComponentClass.Other);
		}
		[Test]
		public void ValidateUnits_Other_Fail()
		{
			this.ValidateUnitFail(Common.ComponentClass.Other);
		}
		
				
		private void ValidateUnitPass(Common.ComponentClass componentClass)
		{
			this.RefreshData = true;

			System.Guid patientGuid = this.CreatePatient();

			UnitTestUtil.RunSQL(_cleanupSQL);
			this.SetUpData(patientGuid, componentClass, true, true, "R", "R", true);
			BOL.BloodUnit bloodUnit = null;
			
			switch (componentClass)
			{
				case Common.ComponentClass.RBC:
					bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291100", Common.ABO.A, Common.RH.Positive, componentClass);
					bloodUnit.DonationType.DonationTypeCode = Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForAutologousUseOnly);
					bloodUnit.DirectedPatient = new BOL.Patient(patientGuid);
					break;
				case Common.ComponentClass.FFP:
					bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E010", "E0749100", Common.ABO.A, Common.RH.Positive, componentClass);
					bloodUnit.DonationType.DonationTypeCode = Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForAutologousUseOnly);
					bloodUnit.DirectedPatient = new BOL.Patient(patientGuid);
					break;
				case Common.ComponentClass.CRYO:
					bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E028", "E3573100", Common.ABO.A, Common.RH.Positive, componentClass);
					bloodUnit.DonationType.DonationTypeCode = Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForAutologousUseOnly);
					bloodUnit.DirectedPatient = new BOL.Patient(patientGuid);
					break;
				case Common.ComponentClass.PLT:
					bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E020", "E2792100", Common.ABO.A, Common.RH.Positive, componentClass);
					bloodUnit.DonationType.DonationTypeCode = Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForAutologousUseOnly);
					bloodUnit.DirectedPatient = new BOL.Patient(patientGuid);
					break;
				case Common.ComponentClass.Other:
					bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E034", "E3649100", Common.ABO.A, Common.RH.Positive, componentClass);
					bloodUnit.DonationType.DonationTypeCode = Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForAutologousUseOnly);
					bloodUnit.DirectedPatient = new BOL.Patient(patientGuid);
					break;
				case Common.ComponentClass.WB:
					bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E001", "E0001100", Common.ABO.A, Common.RH.Positive, componentClass);
					bloodUnit.DonationType.DonationTypeCode = Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForAutologousUseOnly);
					bloodUnit.DirectedPatient = new BOL.Patient(patientGuid);
					break;
			}
		
			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + _orderedComponent.PatientGuid + "'");
		
			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			if (bloodUnit.IsFrozen)
			{
				// Frozen units, even if perfect, will always have one message (the one
				// that warns that the unit must be thawed prior to issue)
				Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 1);
			}
			else
			{
				Assert.IsTrue(_patientUnitSelection.ValidationMessages.Count == 0);
			}
		}


		private void ValidateUnitFail(Common.ComponentClass componentClass)
		{
			#region setup

			System.Guid patientGuid = this.CreatePatient();

			// clean up old order first
			UnitTestUtil.RunSQL(_cleanupSQL);
			this.SetUpData(patientGuid, componentClass, true, true, "R", "R", true);

			BOL.BloodUnit bloodUnit = null;
			
			switch (componentClass)
			{
				case Common.ComponentClass.RBC:
					bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, componentClass);
					break;
				case Common.ComponentClass.FFP:
					bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E010", "E0749V00", Common.ABO.A, Common.RH.Positive, componentClass);
					break;
				case Common.ComponentClass.CRYO:
					bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E028", "E3573V00", Common.ABO.A, Common.RH.Positive, componentClass);
					break;
				case Common.ComponentClass.PLT:
					bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E020", "E2792V00", Common.ABO.A, Common.RH.Positive, componentClass);
					break;
				case Common.ComponentClass.Other:
					bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E034", "E3649V00", Common.ABO.A, Common.RH.Positive, componentClass);
					break;
				case Common.ComponentClass.WB:
					bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E001", "E0001V00", Common.ABO.A, Common.RH.Positive, componentClass);
					break;
			}
			
			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + _orderedComponent.PatientGuid + "'");
			
			#endregion
		

			#region Not Overridable

			bool success = false;

			// Invalid unit status
			string sql = "UPDATE bloodunitstatus SET discardcannedcommentguid = (SELECT TOP 1 cannedcommentguid FROM cannedcomment where divisioncode = '" + UnitTestConfig.TestDivisionCode + "' AND cannedcommentcategorycode = 'QD') WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL(sql);
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);
		
			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BadStatus, _patientUnitSelection.ValidationMessages);
			Assert.IsTrue(success, "Invalid status");
            			
			// Reset
			sql = "UPDATE BloodUnitStatus SET DiscardCannedCommentGuid = null WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL(sql);
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);

			

			// Now, test w/ issued
			Guid issuedUnitGuid = Guid.NewGuid();
			DataCreator.InsertIssuedUnit(issuedUnitGuid, bloodUnit.BloodUnitGuid);
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);
			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR344NotInBloodBank, _patientUnitSelection.ValidationMessages);
			Assert.IsTrue(success, "Invalid status");

			sql = "DELETE FROM IssuedUnit WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL(sql);
			sql = "UPDATE BloodUnitStatusCodeCurrent SET UnitStatusCode = 'A' WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'";
			UnitTestUtil.RunSQL(sql);
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);
			


			// BR_3.33, TT_3.04: Incompatible product types
			string productTypeCodeBefore = bloodUnit.ProductType.ProductTypeCode;
			if (componentClass == Common.ComponentClass.FFP) bloodUnit.ProductType.ProductTypeCode = "E001";
			else bloodUnit.ProductType.ProductTypeCode = "E010";
			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR333InvalidProductType, _patientUnitSelection.ValidationMessages);
			Assert.IsTrue(success, "BR_333 incompatible product types");
			// Reset
			bloodUnit.ProductType.ProductTypeCode = productTypeCodeBefore;

			// Emergency issue:
			// set specimen to unacceptable to make EI
			BOL.Specimen specimen = _orderedComponent.GetSpecimen();
			_orderedComponent.AssociateWithSpecimen(null); 
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));

			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.MustEmergencyIssue, _patientUnitSelection.ValidationMessages);
			if (componentClass == Common.ComponentClass.RBC ||
				componentClass == Common.ComponentClass.WB )
				Assert.IsTrue(success, "Emergency issue");
			// Reset:
			_orderedComponent.AssociateWithSpecimen(specimen);
			_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));



			// BR_3.16: ABORh confirmation 
			bloodUnit.BloodUnitMedia.AboRh.RH = Common.RH.Negative;

			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR316ABORhConfirmation, _patientUnitSelection.ValidationMessages);
			if (new BOL.Division(_orderedComponent.DivisionCode).ServiceTypeIndicator == true
				&& (bloodUnit.ProductType.AboRhConfirmationRequired))
			{
				Assert.IsTrue(success, "BR_316 ABO/Rh confirmation");
			}
			else
			{
				Assert.IsTrue(!success, "BR_316 ABO/Rh confirmation");
			}
			// Reset:
			bloodUnit.BloodUnitMedia.AboRh.RH = Common.RH.Positive;

			// ToDo:
			// BR_3.28, BR_3.19: Already assigned to different patient
			if (componentClass == Common.ComponentClass.RBC)
			{
				Hashtable guids = DataCreator.CreateCrossmatchedBloodUnitTestData();
				Guid unitGuid = (Guid) guids["bloodUnitGuid"];
				BOL.BloodUnit assignedUnit = new BOL.BloodUnit(unitGuid);
				_patientUnitSelection.ValidateUnit(assignedUnit, DateTime.Now);
				success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR328AssignedToAnotherPatient, _patientUnitSelection.ValidationMessages);
				Assert.IsTrue(success, "328 already assigned");
			}


			// BR_3.28, BR_3.19: Already assigned to same patient
			Guid orderedUnitGuid = Guid.Empty;
			if (componentClass == Common.ComponentClass.RBC || componentClass == Common.ComponentClass.WB)
			{
				orderedUnitGuid = SetUpOrderedUnitData(bloodUnit.BloodUnitGuid, _orderedComponent.OrderedComponentGuid, true, "C");
			}
			else
			{
				orderedUnitGuid = SetUpOrderedUnitData(bloodUnit.BloodUnitGuid, _orderedComponent.OrderedComponentGuid, false, null);
			}
			//must recreate the unit to get the new status:
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);

			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.AlreadySelectedForThisPatient, _patientUnitSelection.ValidationMessages);
			Assert.IsTrue(success, "328 already assigned to this patient");
			// Reset:
			UnitTestUtil.RunSQL("DELETE FROM OrderedUnit WHERE OrderedUnitGuid = '" + orderedUnitGuid+ "'");
			//must recreate the unit to get the old status:
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);



			if (componentClass == Common.ComponentClass.RBC)
			{
				// BR_3.19 - Different divisions
				string divisionCodeBefore = bloodUnit.DivisionCode;
				string divisionCode = (string) UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 DivisionCode FROM VamcDivision WHERE DivisionCode != '" + UnitTestConfig.TestDivisionCode + "'")["DivisionCode"];
				Common.LogonUser.LogonUserDivisionCode = divisionCode;
				BOL.BloodUnit unitDiv = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", bloodUnit.BloodUnitMedia.AboRh.Abo, bloodUnit.BloodUnitMedia.AboRh.RH, (Common.ComponentClass) bloodUnit.BloodComponentClassID, divisionCode);
				_patientUnitSelection.ValidateUnit(unitDiv, DateTime.Now);
				success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR344NotInBloodBank, _patientUnitSelection.ValidationMessages);
				Assert.IsTrue(success, "344 Unit in other division");
				Common.LogonUser.LogonUserDivisionCode = UnitTestConfig.TestDivisionCode;
			}

			
			// BR_3.13 - Already crossmatched
			// must create another set of orders
			this.SetUpData(patientGuid, componentClass, true, true, "R", "R", false);
			orderedUnitGuid = this.SetUpOrderedUnitData(bloodUnit.BloodUnitGuid, _orderedComponentTwo.OrderedComponentGuid, true, "C");

			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.AlreadySelectedForThisPatient, _patientUnitSelection.ValidationMessages);
			Assert.IsTrue(success, "313 already crossmatched");
			UnitTestUtil.RunSQL("DELETE FROM OrderedUnit WHERE OrderedUnitGuid = '" + orderedUnitGuid+ "'");
			UnitTestUtil.RunSQL(_cleanupSQLTwo);
			


			// BR_3.43: Unit quarantined
			bloodUnit.BloodUnitStatus.QuarantineIndicator = true;
			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR343UnitQuarantined, _patientUnitSelection.ValidationMessages);
			Assert.IsTrue(success, "BR_343 quarantine");
			// Reset:
			bloodUnit.BloodUnitStatus.QuarantineIndicator = false;

			/*
			if (componentClass == Common.ComponentClass.RBC)
			{
				
				// Setup 
				//Guid shipmentGuid = (Guid) UnitTestUtil.RunSQLGetDataRow("SELECT top 1 ShipmentGuid FROM Shipment WHERE DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' AND RecordStatusCode = 'A'")["ShipmentGuid"];
				//UnitTestUtil.RunSQL("DELETE FROM BloodUnitShipment WHERE IncomingShipmentGuid = '" + shipmentGuid + "'");
				Guid shipmentGuid = Guid.NewGuid();
				sql = "INSERT INTO Shipment(ShipmentGuid, InvoiceNumber, ShipmentDirection, DivisionCode, RecordStatusCode, LastUpdateDate, LastUpdateUser, LastUpdateFunctionId) " +
					"VALUES ('" + shipmentGuid + "','111',0,'" + UnitTestConfig.TestDivisionCode +  "','A','" + DateTime.Now.ToString() + "','" + Common.LogonUser.LogonUserName + "',9999)";
				UnitTestUtil.RunSQL(sql);

				sql = "INSERT INTO BloodUnitShipment (BloodUnitShipmentGuid, BloodUnitGuid, OutgoingShipmentGuid, RecordStatusCode, LastUpdateDate, LastUpdateUser, LastUpdateFunctionID) VALUES " +
															"('" + Guid.NewGuid() + "','" + bloodUnit.BloodUnitGuid + "','" + shipmentGuid + "','A','" + DateTime.Now.ToString() + "','" + Common.LogonUser.LogonUserName + "',9999)";
				UnitTestUtil.RunSQL(sql);
//
//
			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
				success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR345OutgoingShipment, _patientUnitSelection.ValidationMessages);
				Assertion.IsTrue("BR_345 Outgoing shipment", success);

				// Clean up
				UnitTestUtil.RunSQL("DELETE FROM BloodUnitShipment WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'");
			}
			*/






			// BR_3.46: Restricted for a different patient
			char donationTypeCodeBefore = bloodUnit.DonationType.DonationTypeCode;
			bloodUnit.DonationType.DonationTypeCode = Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForAutologousUseOnly);
			bloodUnit.DirectedPatient.PatientGuid = Guid.NewGuid();
			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR346UnitRestricted, _patientUnitSelection.ValidationMessages);
			Assert.IsTrue(success, "BR_346 Restricted for different patient");
			// Reset
			bloodUnit.DonationType.DonationTypeCode = donationTypeCodeBefore;
			bloodUnit.DirectedPatient = null;

			
			#endregion


			#region Overridable

			// BR_3.23: Incompatible results exist: ToDo


			// BR_3.06, BR_3.15
			if (componentClass == Common.ComponentClass.WB || componentClass == Common.ComponentClass.RBC)
			{
				this.SetUpData(patientGuid, componentClass, true, true, "R", "R", true);
				this.SetupAntigenTr(_orderedComponent.Patient, bloodUnit.BloodUnitGuid);
				_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
				success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR315UnitNotTyped, _patientUnitSelection.ValidationMessages);
				Assert.IsTrue(success, "BR_315 Unit not tested for antigen");
				// Clean up
				UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + _orderedComponent.PatientGuid + "'");
			}


			if (componentClass != Common.ComponentClass.PLT && componentClass != Common.ComponentClass.Other)
			{
				// BR_3.10: Incompatible units
				BOL.BloodUnit bloodUnit310 = null;
				switch (componentClass)
				{
					case Common.ComponentClass.RBC:
						bloodUnit310 = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.B, Common.RH.Positive, componentClass);
						break;
					case Common.ComponentClass.FFP:
						bloodUnit310 = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E010", "E0749V00", Common.ABO.B, Common.RH.Positive, componentClass);
						break;
					case Common.ComponentClass.CRYO:
						bloodUnit310 = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E028", "E3573V00", Common.ABO.I, Common.RH.Positive, componentClass);
						break;
					case Common.ComponentClass.WB:
						bloodUnit310 = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E001", "E0001V00", Common.ABO.B, Common.RH.Positive, componentClass);
						break;
				}
				bloodUnit310 = new BOL.BloodUnit(bloodUnit310.BloodUnitGuid);

				_patientUnitSelection.ValidateUnit(bloodUnit310, DateTime.Now);
				success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR310Override, _patientUnitSelection.ValidationMessages);
				Assert.IsTrue(success, "BR_310 Incompatible");
			}


			//			// BR_3.08: Anti-D
			//			_orderedComponent.Patient.AboRh.RH = Common.RH.Negative;
			//			this.SetAntiD(_orderedComponent.Patient, bloodUnit.BloodUnitGuid);
			//			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			//			success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR306, _patientUnitSelection.ValidationMessages);
			//			Assertion.IsTrue("BR_308 Anti D", success);
			//			// Reset
			//			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + _orderedComponent.PatientGuid + "'");
			//			_orderedComponent.Patient.AboRh.RH = Common.RH.Positive;


			// BR_3.19: Expired unit
			DateTime expirationBefore = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			bloodUnit.BloodUnitMedia.UnitExpirationDate = System.DateTime.Now.AddDays(-1);
			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);		
			success = VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR319UnitExpired, _patientUnitSelection.ValidationMessages);
			Assert.IsTrue(success, "319 Unit expired" );
			// Reset:
			bloodUnit.BloodUnitMedia.UnitExpirationDate = expirationBefore;


			// BR_3.21: Rh requirement
			if (componentClass == Common.ComponentClass.RBC || componentClass == Common.ComponentClass.WB)
			{
				_orderedComponent.Patient.AboRh.RH = Common.RH.Negative;
				_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
				success = VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR321, _patientUnitSelection.ValidationMessages);
				Assert.IsTrue(success, "321 Unit Rh" );
				// reset data
				_orderedComponent.Patient.AboRh.RH = Common.RH.Positive;
			}


			// BR_3.14: TRs
			if (componentClass != Common.ComponentClass.CRYO && componentClass != Common.ComponentClass.FFP)
			{
				this.SetupAllCRs(_orderedComponent.Patient);
				_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
				success = VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR314TransfusionRequirements, _patientUnitSelection.ValidationMessages);
				Assert.IsTrue(success, "314 Transfusion requirements" );
				UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + _orderedComponent.PatientGuid + "'");
			}


			// BR_3.34: Units chosen in correct order
			BOL.BloodUnit bloodUnitAutologous = null;
			BOL.BloodUnit bloodUnitDirected = null;
			switch (componentClass)
			{
				case Common.ComponentClass.RBC:
					bloodUnitAutologous = CreateBloodUnit("E002" ,"E0291V00", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.RBC);
					bloodUnitDirected = CreateBloodUnit("E002" ,"E0291V00", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.RBC);
					break;
				case Common.ComponentClass.WB:
					bloodUnitAutologous = CreateBloodUnit("E001", "E0001V00", Common.ABO.A, Common.RH.Positive, componentClass);
					bloodUnitDirected = CreateBloodUnit("E001", "E0001V00", Common.ABO.A, Common.RH.Positive, componentClass);
					break;
			}
			// test with RBC and WB will cover all conditions
			if (componentClass == Common.ComponentClass.RBC || componentClass == Common.ComponentClass.WB)
			{
				bloodUnit = CreateBloodUnit("E002" ,"E0291V00", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.RBC);
			
				// AutologousPut in transaction
				UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientUnitSelection.OrderedComponent.Patient.PatientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");
				string donate = "UPDATE BloodUnit SET DonationTypeId = " + (int) Common.DonationType.ForAutologousUseOnly + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'";
				UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 10 + " WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");

				// Directed
				UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET DirectedPatientGuid = '" + _patientUnitSelection.OrderedComponent.Patient.PatientGuid + "' WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");
				UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = " + 14 + " WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");

				// Regenerate BUs so they pick up the update donation codes
				bloodUnitAutologous = new BOL.BloodUnit(bloodUnitAutologous.BloodUnitGuid);
				bloodUnitDirected = new BOL.BloodUnit(bloodUnitDirected.BloodUnitGuid);

				//Wipe out the Current Shipiment Record -- to avoid dup inserts
				UnitTestUtil.RunSQL("DELETE FROM BloodUnitCurrentShipment WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid.ToString() + "'");
				// Get data for BloodUnitCurrentShipment table -autologous
				DataRow dtBus = UnitTestUtil.RunSQLGetDataRow("SELECT * FROM BloodUnitShipment WHERE BloodUnitGuid = '" + bloodUnitAutologous.BloodUnitGuid + "'");
				sql = "INSERT INTO BloodUnitCurrentShipment (" +
					"BloodUnitCurrentShipmentGuid, " + 
					"BloodUnitGuid, " +
					"IncomingShipmentGuid, " + 
					"LastUpdateDate, " + 
					"LastUpdateUser, " +
					"LastUpdateFunctionId) VALUES ('" +
					Guid.NewGuid() + "','" + 
					bloodUnitAutologous.BloodUnitGuid + "','" +
					(Guid) dtBus["IncomingShipmentGuid"] + "','" +
					DateTime.Now.ToString() + "','" +
					"Foo" + "'," +
					9999 + ")";
				UnitTestUtil.RunSQL(sql);

				// Get data for BloodUnitCurrentShipment table -directed
				//Wipe out the Current Shipiment Record -- to avoid dup inserts
				UnitTestUtil.RunSQL("DELETE FROM BloodUnitCurrentShipment WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid.ToString() + "'");

				dtBus = UnitTestUtil.RunSQLGetDataRow("SELECT * FROM BloodUnitShipment WHERE BloodUnitGuid = '" + bloodUnitDirected.BloodUnitGuid + "'");
				sql = "INSERT INTO BloodUnitCurrentShipment (" +
					"BloodUnitCurrentShipmentGuid, " + 
					"BloodUnitGuid, " +
					"IncomingShipmentGuid, " + 
					"LastUpdateDate, " + 
					"LastUpdateUser, " +
					"LastUpdateFunctionId) VALUES ('" +
					Guid.NewGuid() + "','" + 
					bloodUnitDirected.BloodUnitGuid + "','" +
					(Guid) dtBus["IncomingShipmentGuid"] + "','" +
					DateTime.Now.ToString() + "','" +
					"Foo" + "'," +
					9999 + ")";
				UnitTestUtil.RunSQL(sql);



				// First, add directed unit when autologous are available
				_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));
				_patientUnitSelection.ValidateUnit(bloodUnitDirected, DateTime.Now);

				success = VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR334RestrictiveUnits, _patientUnitSelection.ValidationMessages);
				Assert.IsTrue(success, "334 More restrictive units available" );


				Assert.IsTrue(!_patientUnitSelection.AllAutologousUnitsChosen());
				Assert.IsTrue(!_patientUnitSelection.AllDirectedUnitsChosen());
				//Assert.IsTrue(_patientUnitSelection.AllRestrictedUnitsChosen());
				


				// Now, try to add a normal unit when autologous are available
				_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));
				_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);

				success = VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR334RestrictiveUnits, _patientUnitSelection.ValidationMessages);
				Assert.IsTrue(success, "334 More restrictive units available" );

				
				Assert.IsTrue(!_patientUnitSelection.AllAutologousUnitsChosen());
				Assert.IsTrue(!_patientUnitSelection.AllDirectedUnitsChosen());
				//Assert.IsTrue(_patientUnitSelection.AllRestrictedUnitsChosen());



			
				// Now, try to add a restricted unit when directed/ autologous are available
				_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, this.GetVbecsUser(Common.UserRoleType.Technologist));
				//				BOL.Patient patient = new BOL.Patient();
				//				patient.PatientGuid = Guid.NewGuid();
				bloodUnit.DirectedPatient = _orderedComponent.Patient;
				_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);

				success = VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR334RestrictiveUnits, _patientUnitSelection.ValidationMessages);
				Assert.IsTrue(success, "334 More restrictive units available" );		
	
				Assert.IsTrue(!_patientUnitSelection.AllAutologousUnitsChosen());
				Assert.IsTrue(!_patientUnitSelection.AllDirectedUnitsChosen());
				//Assert.IsTrue(_patientUnitSelection.AllRestrictedUnitsChosen());
			}

			#endregion
			

			#region informational

			if (componentClass == Common.ComponentClass.RBC)
			{
				// BR_3.11: Current specimen must be tested:
				this.SetUpData(_orderedComponent.PatientGuid, Common.ComponentClass.RBC, true, true, "R", "R", false);
				_patientUnitSelectionTwo = new BOL.PatientUnitSelection(_orderedComponentTwo, GetVbecsUser(Common.UserRoleType.Technologist));
				UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE PatientSpecimenGuid = '" + _orderedComponentTwo.GetSpecimen().SpecimenGuid + "'");
				_patientUnitSelectionTwo.ValidateUnit(bloodUnit, DateTime.Now);
				success = VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR311NoABORhResults, _patientUnitSelectionTwo.ValidationMessages);
				Assert.IsTrue(success, "311 No ABO/Rh results" );
				// Clean up
				UnitTestUtil.RunSQL(_cleanupSQLTwo);
				UnitTestUtil.RunSQL("DELETE FROM OrderedComponent WHERE OrderedComponentGuid = '" + _orderedComponent.OrderedComponentGuid + "'");
			}


			// BR_3.39 - Biohazardous indicator
			bloodUnit.BloodUnitStatus.BiohazardousWarningIndicator = true;
			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR339BiohazardousWarningIndicator, _patientUnitSelection.ValidationMessages);
			Assert.IsTrue(success, "Br_339 test");
			// Reset
			bloodUnit.BloodUnitStatus.BiohazardousWarningIndicator = false;

			// BR_3.12
			switch (componentClass)
			{
				case Common.ComponentClass.WB:
				{
					bloodUnit.ProductType.ProductTypeCode = "E004";
					break;
				}
				case Common.ComponentClass.RBC:
				{
					bloodUnit.ProductType.ProductTypeCode = "E004";
					break;
				}
				case Common.ComponentClass.FFP:
				{
					bloodUnit.ProductType.ProductTypeCode = "E010";
					break;
				}
				case Common.ComponentClass.CRYO:
				{
					bloodUnit.ProductType.ProductTypeCode = "E028";
					break;
				}
					// other can never be frozen
				case Common.ComponentClass.Other:
				{
					bloodUnit.ProductType.ProductTypeCode = "E040";
					break;
				}
				case Common.ComponentClass.PLT:
				{
					bloodUnit.ProductType.ProductTypeCode = "E025";
					break;
				}
			}
			_patientUnitSelection.ValidateUnit(bloodUnit, DateTime.Now);
			success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR312UnitFrozen, _patientUnitSelection.ValidationMessages) ||
				this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR312UnitFrozenEmergencyIssue, _patientUnitSelection.ValidationMessages);
			if (componentClass == Common.ComponentClass.Other)
			{
				Assert.IsTrue(!success, "Br_312 test");	
			}
			else
			{
				Assert.IsTrue(success, "Br_312 test");
			}


			// BR_3.52
			if (componentClass == Common.ComponentClass.FFP)
			{
				UnitTestUtil.RunSQL("UPDATE Patient SET AboRhChangeIndicator = 1 WHERE PatientGuid = '" + _orderedComponent.PatientGuid + "'");
				Guid orderedComponentGuid = this.CreateTestData(_orderedComponent.PatientGuid, Common.ComponentClass.FFP, false, false, "N", "R", false);

				DataRow dtRow = UnitTestUtil.RunSQLGetDataRow("SELECT * FROM OrderedComponent WHERE OrderedComponentGuid = '" + orderedComponentGuid + "'");
				dtRow.Table.Columns.Add("PatientGuid", typeof(System.Guid));
				dtRow["PatientGuid"] = _orderedComponent.PatientGuid;

				_patientUnitSelectionTwo = new BOL.PatientUnitSelection(new BOL.OrderedComponent(dtRow), GetVbecsUser(Common.UserRoleType.Technologist));
				_patientUnitSelectionTwo.ValidateUnit(bloodUnit, DateTime.Now);
				success = this.VerifyValidationMessage(Common.UC03UnitSelectErrorMessages.BR352JustifiedABORh, _patientUnitSelectionTwo.ValidationMessages); 
				Assert.IsTrue(success, "BR352");
				// Clean up
				UnitTestUtil.RunSQL("UPDATE Patient SET AboRhChangeIndicator = 0 WHERE PatientGuid = '" + _orderedComponent.PatientGuid + "'");
				UnitTestUtil.RunSQL(_cleanupSQLTwo);
			}

			#endregion
		}

		#endregion


		#endregion




		



		

		


		

		


		

		


		

		[Test]
		public void DisplayOptionalCrossmatch_True()
		{
			_genericBU.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisMonocytes);
			bool display = this._patientUnitSelection.DisplayOptionalCrossmatch(_genericBU);
			Assert.IsTrue(display);
		}
		[Test]
		public void DisplayOptionalCrossmatch_False()
		{
			_genericBU.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.RedBloodCells);
			bool display = this._patientUnitSelection.DisplayOptionalCrossmatch(_genericBU);
			Assert.IsTrue(!display);	
		}


		



		//		[Test]
		//		public void DisplayOptionalCrossmatch_Pass()
		//		{
		//			bool display = false;
		//
		//			this._bloodUnit.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisPlatelets);
		//			display = this._patientUnitSelection.DisplayOptionalCrossmatch(this._bloodUnit);
		//			Assertion.IsTrue(display == true);
		//
		//			this._bloodUnit.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.Granulocytes);
		//			display = this._patientUnitSelection.DisplayOptionalCrossmatch(this._bloodUnit);
		//			Assertion.IsTrue(display == true);
		//
		//			this._bloodUnit.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisGranulocytes);
		//			display = this._patientUnitSelection.DisplayOptionalCrossmatch(this._bloodUnit);
		//			Assertion.IsTrue(display == true);
		//
		//			this._bloodUnit.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisGranulocytesPlatelets);
		//			display = this._patientUnitSelection.DisplayOptionalCrossmatch(this._bloodUnit);
		//			Assertion.IsTrue(display == true);
		//
		//			this._bloodUnit.ProductType.ProductTypeCode = Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisFreshFrozenPlasma);
		//			display = this._patientUnitSelection.DisplayOptionalCrossmatch(this._bloodUnit);
		//			Assertion.IsTrue(display == false);
		//		}

		#endregion


		#region helper methods

		private void SetupAllCRs(BOL.Patient patient)
		{
			DataTable dtInsertSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(false);
			DataTable dtUpdateSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(true);
			DataTable dtInsertTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);
			DataTable dtUpdateTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(true);

			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + patient.PatientGuid + "'");
		
			// Irradiated
			BOL.PatientTransfusionRequirement pTR = new BOL.PatientTransfusionRequirement(patient, "Irradiated cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			Common.ComponentRequirement req = Common.ComponentRequirement.IRRADIATE;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// Leuko
			pTR = new BOL.PatientTransfusionRequirement(patient, "Leuko-reduced cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.LEUKORED;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// Washed RBC
			pTR = new BOL.PatientTransfusionRequirement(patient, "Washed RBC products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.WASHEDRBC;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// Washed PLT
			pTR = new BOL.PatientTransfusionRequirement(patient, "Washed PLT products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.WASHEDPLT;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// Sickle cell
			pTR = new BOL.PatientTransfusionRequirement(patient, "Sickle Cell Negative RBC products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.SCNEG;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// CMV Neg
			pTR = new BOL.PatientTransfusionRequirement(patient, "CMV Negative cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.CMVNEG;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);
		
			bool success = BOL.Patient.SaveSIsAndTRs(dtInsertSpecialInstructions, dtUpdateSpecialInstructions, dtInsertTransfusionRequirements, dtUpdateTransfusionRequirements, Common.UpdateFunction.UnitTests);
			Assert.IsTrue(success);
		}

		private void AddDataRow(DataTable dtCrs, Common.ComponentRequirement req, BOL.PatientTransfusionRequirement pTR, BOL.Patient patient)
		{
			DataRow drTRs = dtCrs.NewRow();
			drTRs[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = System.Guid.NewGuid();
			drTRs[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drTRs[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = pTR.TransfusionRequirement;
			drTRs[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( pTR.TransfusionRequirementCategoryCode );
			drTRs[TABLES.PatientTransfusionRequirement.CreateDate] = pTR.CreateDate;
			drTRs[TABLES.PatientTransfusionRequirement.CreateUser] = pTR.CreateUser;
			drTRs[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = pTR.CreateDivisionCode;
			drTRs[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			//drTRs[TABLES.PatientTransfusionRequirement.ComponentRequirementId] = pTR.ComponentRequirementId;
			drTRs[TABLES.PatientTransfusionRequirement.ComponentRequirementId] = (int) req;
			drTRs[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drTRs[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drTRs[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtCrs.Rows.Add(drTRs);
		}

		private Guid SetUpOrderedUnitData(Guid bloodUnitGuid, 
			Guid orderedComponentGuid, 
			bool crossmatchRequired, 
			string crossmatchResultCode)
		{
			Guid orderedUnitGuid = Guid.NewGuid();

			string insertOrderedUnit = "INSERT INTO ORDEREDUNIT (";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedComponentGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.BloodUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchRequiredIndicator + ",";
			if (crossmatchResultCode != null)
			{
				insertOrderedUnit += TABLES.OrderedUnit.CrossmatchResultCode + ",";
				insertOrderedUnit += TABLES.OrderedUnit.CrossmatchTechId + ",";
			}
			insertOrderedUnit += TABLES.OrderedUnit.RecordStatusCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.DivisionCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateUser + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			insertOrderedUnit += "VALUES (";
			insertOrderedUnit += "'" + orderedUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + _orderedComponent.OrderedComponentGuid + "',";
			insertOrderedUnit += "'" + bloodUnitGuid + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now) + "',";
			if (crossmatchRequired)	insertOrderedUnit += "1,";
			else insertOrderedUnit += "0,";
			if (crossmatchResultCode != null)
			{
				insertOrderedUnit += "'" + crossmatchResultCode + "',";
				insertOrderedUnit += "'" + Environment.UserName + "',";
			}
			insertOrderedUnit += "'A',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(insertOrderedUnit.ToString());
			return orderedUnitGuid;
		}

		private void SetUpData(Guid patientGuid, 
			Common.ComponentClass componentClass, 
			bool specimenRequired, 
			bool associateWithSpecimen,
			string specimenStatusCode,
			string orderUrgencyCode,
			bool originalOrder)
		{
			// Create all of the background data for an RBC
			Guid orderedComponentGuid = this.CreateTestData(patientGuid, componentClass, specimenRequired, associateWithSpecimen, specimenStatusCode, orderUrgencyCode, originalOrder );
				
			DataRow dtRow = UnitTestUtil.RunSQLGetDataRow("SELECT * FROM OrderedComponent WHERE OrderedComponentGuid = '" + orderedComponentGuid + "'");
			dtRow.Table.Columns.Add("PatientGuid", typeof(System.Guid));
			dtRow["PatientGuid"] = patientGuid;

			if (originalOrder)
			{
				_orderedComponent = new BOL.OrderedComponent(dtRow);
				_patientUnitSelection = new BOL.PatientUnitSelection(_orderedComponent, GetVbecsUser(Common.UserRoleType.Technologist));
				_genericBU = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			}
			else
			{
				_orderedComponentTwo = new BOL.OrderedComponent(dtRow);
			}
		}

		private BOL.VbecsUser GetVbecsUser(Common.UserRoleType userRole)
		{
            DataRow dtRow = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 VbecsUser.VbecsUserId, VU.DivisionCode FROM VbecsUser JOIN VbecsUserDivisionRole VU ON VbecsUser.VbecsUserId = VU.VbecsUserId WHERE VbecsUser.SystemUserIndicator = 0 AND VbecsUser.RecordStatusCode = 'A' AND DivisionCode = '" + _orderedComponent.DivisionCode + "'");
			string updateSql = "UPDATE VbecsUserDivisionRole SET " + TABLES.VbecsUserDivisionRole.UserRoleId + " = " + (int) userRole + ", RecordStatusCode = 'A', LastUpdateDate = '" + DateTime.Now + "' WHERE " + TABLES.VbecsUserDivisionRole.DivisionCode + " = '" + _orderedComponent.DivisionCode + "' AND " + TABLES.VbecsUserDivisionRole.VbecsUserId + " = '" + (string) dtRow[TABLES.VbecsUserDivisionRole.VbecsUserId] + "'";
			UnitTestUtil.RunSQL(updateSql);
			BOL.VbecsUser vbecsUser =  new BOL.VbecsUser( (string) dtRow[TABLES.VbecsUserDivisionRole.VbecsUserId] );
			vbecsUser.LogonToDivision( _orderedComponent.DivisionCode );
			vbecsUser.CurrentDivisionRole.Division.ServiceTypeIndicator = true;
			return vbecsUser;
		}

		// Needed to test the error message coming back
		private bool HasAuthorityToIssueABOIncompatibleBlood(BOL.VbecsUser user)
		{
			return user.IsAuthorizedForFunction( Common.FunctionCode.SelectIssueABOIncompatibleblood );
		}
		private bool HasAuthorityToIssueAntigenPositiveOrUntypedRBCs(BOL.VbecsUser user)
		{
			return user.IsAuthorizedForFunction( Common.FunctionCode.IssueAntigenPosOrUntypedRedbloodcells );
		}

		private bool AddUnit(BOL.BloodUnit bloodUnit)
		{
			Random random = new Random();

			_patientUnitSelection.OrderedComponent.Patient.TransfusionRequirements.Rows.Clear();
			_patientUnitSelection.OrderedComponent.Patient.ClinicallySignificantAntibodies.Rows.Clear();
		
			return _patientUnitSelection.SelectedUnitsCollection.AddUnit(bloodUnit, DateTime.Now, UnitTestConfig.User_Technologist, Common.UserRoleType.EnhancedSupervisor);
		}

		
		/// <summary>
		/// This verifies that that correct error message has been generated.
		/// </summary>
		/// <param name="message"></param>
		/// <param name="messages"></param>
		/// <returns></returns>
		private bool VerifyValidationMessage(Common.UC03UnitSelectErrorMessages message, ArrayList messages)
		{
			for (int i=0; i<messages.Count; i++)
			{
				BOL.PatientUnitSelectionMessage pUSM = (BOL.PatientUnitSelectionMessage) messages[i];
				if (pUSM.Message == message && pUSM.MessageText.IndexOf("exception") == -1)
				{
					return true;
				}
			}
			return false;
		}

		private Guid SetupSpecimenTest(Guid specimenGuid, Guid orderedTestGuid, string test, string result)
		{
			Thread.Sleep(1000);

			StringBuilder query = new StringBuilder();
			Guid specimenTestGuid = Guid.NewGuid();
			
			System.DateTime dateTested = System.DateTime.Now;

			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			if (result != null)	query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestGuid).Append("',");
			query.Append("'").Append(specimenGuid).Append("',");
			query.Append("'").Append(orderedTestGuid).Append("',");
			query.Append("'" + test + "',");
			if (result != null) query.Append("'"+ result +"',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(dateTested)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			UnitTestUtil.RunSQL(query.ToString());

			return specimenTestGuid;
		}

		private Guid CreatePatient()
		{
			Guid patientGuid = Guid.NewGuid();
			StringBuilder query = new StringBuilder();
			// Patient
			query.Append("INSERT INTO PATIENT (");
			query.Append(TABLES.Patient.PatientGuid + ",");
			query.Append(TABLES.Patient.PatientSsn + ",");
			query.Append(TABLES.Patient.VistaPatientId + ",");
			query.Append(TABLES.Patient.PatientLastName + ",");
			query.Append(TABLES.Patient.PatientFirstName + ",");
			query.Append(TABLES.Patient.PatientDob + ",");
			query.Append(TABLES.Patient.PatientDobCode + ",");
			query.Append(TABLES.Patient.RecordStatusCode + ",");
			query.Append(TABLES.Patient.LastUpdateDate + ",");
			query.Append(TABLES.Patient.LastUpdateUser + ",");
			query.Append(TABLES.Patient.LastUpdateFunctionId + ")");
			query.Append("VALUES(");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(new Random().Next(11111,99999).ToString() + new Random().Next(1111,9999).ToString()).Append("',");
			query.Append("'").Append(SprocHelper.GetVistaPatientId().ToString()).Append("',");
			query.Append("'CreateXMatchBUTData',");
			query.Append("'SprocHelper',");
			query.Append("'").Append(DateTime.Today.AddYears(-50)).Append("',");
			query.Append("'V',");
			query.Append("'").Append(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserName).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("')");

			UnitTestUtil.RunSQL(query.ToString());
			return patientGuid;
		}


		private Guid CreateTestData(Guid patientGuid, 
			Common.ComponentClass componentClass, 
			bool specimenRequired, 
			bool associateWithSpecimen,
			string specimenStatusCode, // A: In BB,  C: Converted,  N: NSR,  R: Required
			string orderUrgencyCode, // A: ASAP.  P: Pre-Op,  R: Routine,  S: Stat
			bool originalOrder) // Indicates whether this is the order created at the beginning of test setup 
		{
			#region clear patient data

			// Clear any data out of specimen test
			string sql = "DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + patientGuid + "'";
			UnitTestUtil.RunSQL(sql);

			sql = "UPDATE Patient SET BloodTypeCode = NULL, RhFactorCode = NULL WHERE PatientGuid = '" + patientGuid + "'";
			UnitTestUtil.RunSQL(sql);

			sql = "DELETE FROM SpecimenTest WHERE SpecimenTestGuid IN " + 
				"(SELECT SpecimenTestGuid FROM SpecimenTest " +
				"JOIN PatientSpecimen PS ON PS.PatientSpecimenGuid = SpecimenTest.PatientSpecimenGuid " +
				"WHERE PS.PatientGuid =  '" + patientGuid + "')";
			UnitTestUtil.RunSQL(sql);

			#endregion


			#region Create Test Data

			// Patient Treatment
			Guid patientTreatmentTestOneGuid = Guid.NewGuid();
			Guid patientTreatmentTestTwoGuid = Guid.NewGuid();

			// Patient Order
			Guid patientOrderTestOneGuid = Guid.NewGuid();
			Guid patientOrderTestTwoGuid = Guid.NewGuid();

			// Ordered Test
			Guid orderedTestTestOneGuid = Guid.NewGuid();
			Guid orderedTestTestTwoGuid = Guid.NewGuid();
			_orderedTestGuid = orderedTestTestTwoGuid;

			// Patient Specimen
			Guid patientSpecimenTestOneGuid = Guid.NewGuid();
			_specimenOneGuid = patientSpecimenTestOneGuid;
			Guid patientSpecimenTestTwoGuid = Guid.NewGuid();
			
			// Specimen Tests
			Guid specimenTestAboTestOneGuid = Guid.NewGuid();
			Guid specimenTestRhTestOneGuid = Guid.NewGuid();
			Guid specimenTestAboTestTwoGuid = Guid.NewGuid();
			Guid specimenTestRhTestTwoGuid = Guid.NewGuid();
			Guid patientSpecimenTestABScreenGuid = Guid.NewGuid();

			// Ordered Component
			Guid orderedComponentGuid = Guid.NewGuid();
	 
			// First, we need two new Specimen Uids
			DataTable specimenUidTable = UnitTestUtil.RunSQLGetDataTable("SELECT SpecimenUid FROM PatientSpecimen WHERE SpecimenUid IS NOT NULL ORDER BY SpecimenUid DESC");
			DataColumn [] specimenUidKeys = new DataColumn []{ specimenUidTable.Columns[TABLES.PatientSpecimen.SpecimenUid] };
			specimenUidTable.PrimaryKey = specimenUidKeys;
			//
			long maxUid = 999999999999999;
			long minUid = 100000000000000;
			long specimenUid1 = minUid;
			long specimenUid2 = minUid;
			long startIdx = minUid;
			//
			for (long idx = startIdx; idx < maxUid; idx++)
			{
				if( !specimenUidTable.Rows.Contains(new object[] { idx }) )
				{ 
					specimenUid1 = idx;
					startIdx = idx + 1;
					break;
				}
			}
			//
			for (long idx = startIdx; idx < maxUid; idx++)
			{
				if( !specimenUidTable.Rows.Contains(new object[] { idx }) )
				{ 
					specimenUid2 = idx;
					break;
				}
			}
			//
			StringBuilder query = new StringBuilder();
			//
			query.Append("BEGIN TRANSACTION T1 ");
			
			// PatientTreatment
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// First entry into PatientOrder
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Second entry into PatientOrder (is this one needed?)
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// OrderedTest #1
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// OrderedTest #2
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen #1
			query.Append("INSERT INTO PATIENTSPECIMEN (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.MaintainableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(specimenUid1).Append("',");
			query.Append(1 + ",");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen #2
			query.Append("INSERT INTO PATIENTSPECIMEN (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.MaintainableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(specimenUid2).Append("',");
			query.Append(1 + ",");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			DateTime dateTested = System.DateTime.Now;

			// Specimen (#1) test: A
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(dateTested)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen (#1) test: P
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(dateTested)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			Thread.Sleep(5000);	//Bleh, probably not necessary, but, meh
			dateTested = System.DateTime.Now;

			// Specimen (#2) test: A
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(dateTested)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(dateTested)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen (#2) test: P
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(dateTested)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#2) test: Antibody screen
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestABScreenGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'92',");
			query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(dateTested)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// OrderedComponent
			query.Append("INSERT INTO ORDEREDCOMPONENT (");
			query.Append(TABLES.OrderedComponent.OrderedComponentGuid).Append(",");
			query.Append(TABLES.OrderedComponent.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedComponent.CprsOrderNumber).Append(",");
			query.Append(TABLES.OrderedComponent.ComponentClassId).Append(",");
			query.Append(TABLES.OrderedComponent.RequiredUnitQuantity).Append(",");
			query.Append(TABLES.OrderedComponent.RequiredDatetime).Append(",");
			query.Append(TABLES.OrderedComponent.SpecimenRequiredIndicator).Append(",");
			if (associateWithSpecimen) query.Append(TABLES.OrderedComponent.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.OrderedComponent.SpecimenStatusCode).Append(",");
			query.Append(TABLES.OrderedComponent.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedComponent.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedComponent.OrderUrgencyCode).Append(",");
			query.Append(TABLES.OrderedComponent.DivisionCode).Append(",");
			query.Append(TABLES.OrderedComponent.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedComponent.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedComponent.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedComponentGuid.ToString()).Append("',");		// ordered component guid
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");	// patient order guid
            query.Append("'").Append(SprocHelper.GetNewCprsOrderNumber()).Append("',");
			query.Append((int) componentClass + ",");
			query.Append(10 + ",");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			if (specimenRequired) query.Append(1 + ",");
			else query.Append(0 + ",");							// specimen required 
			if (associateWithSpecimen) query.Append("'" + patientSpecimenTestTwoGuid + "',");	// patient specimen
			query.Append("'" + specimenStatusCode + "',");		// specimen status code
			query.Append("'A',");								// order status code
			query.Append("'P',");								// task status code
			query.Append("'" + orderUrgencyCode + "',");// order urgency code
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("COMMIT TRANSACTION T1 ");
			//
			UnitTestUtil.RunSQL(query.ToString());
			query.Remove(0, query.Length);

			#endregion
			
			// Delete test data
			#region Clear Test Data

			query.Append("BEGIN TRANSACTION T2 ");
			query.Append("DELETE FROM ORDEREDCOMPONENT WHERE OrderedComponentGuid ='").Append(orderedComponentGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAboTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestRhTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAboTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestRhTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(patientSpecimenTestABScreenGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTORDER WHERE PatientOrderGuid ='").Append(patientOrderTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTORDER WHERE PatientOrderGuid ='").Append(patientOrderTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientTreatmentGuid ='").Append(patientTreatmentTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientTreatmentGuid ='").Append(patientTreatmentTestTwoGuid).Append("' ").Append("\n");
			query.Append("COMMIT TRANSACTION T2 ");

			if (originalOrder)
			{
				_cleanupSQL = query.ToString();
			}
			else
			{
				_cleanupSQLTwo = query.ToString();
			}
			
			#endregion

			return orderedComponentGuid;
		}

		private void ClearTestData()
		{
			if (_cleanupSQL.Trim() != string.Empty)
			{
				try
				{
					UnitTestUtil.RunSQL(_cleanupSQL);
				}
				catch
				{
					Assert.Fail("Error in ClearTestData():" + _cleanupSQL);
				}
			}
		}

		private void DeleteConflictingTrData(BOL.Patient patient, 
			BOL.BloodUnit bloodUnit,
			bool bloodUnitAntigen,
			bool patientTransfusionRequirement,
			bool bloodUnitTests)
		{
			if (bloodUnitAntigen) UnitTestUtil.RunSQL("DELETE FROM BloodUnitAntigen WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid.ToString() + "'");
			if (patientTransfusionRequirement) UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + patient.PatientGuid.ToString() + "'");
			if (bloodUnitTests) UnitTestUtil.RunSQL("DELETE FROM BloodUnitTest WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'");
		}

		private void SetupBloodUnitTests(Guid bloodUnitGuid, int test, string result)
		{
			DataCreator.InsertBloodUnitTest(Guid.NewGuid(), bloodUnitGuid, Guid.Empty, Guid.Empty, test, result);
		}

		private void SetAntiD(BOL.Patient patient, Guid bloodUnitGuid, bool fromConversion)
		{
			DataTable dtInsertSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(false);
			DataTable dtUpdateSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(true);
			DataTable dtInsertTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);
			DataTable dtUpdateTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(true);
					
			// Create new TR (antibody for Anti-E) 
			BOL.PatientTransfusionRequirement trAntibody = new BOL.PatientTransfusionRequirement(patient, "Anti-D", Common.TransfusionRequirementCategoryCode.AntibodyIdentified, UnitTestConfig.TestDivisionCode);
			DataRow drAntibody = dtInsertTransfusionRequirements.NewRow();

			Guid antibodyGuid = Guid.NewGuid();
			drAntibody[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = antibodyGuid;
			drAntibody[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drAntibody[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = trAntibody.TransfusionRequirement;
			drAntibody[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( trAntibody.TransfusionRequirementCategoryCode );
			drAntibody[TABLES.PatientTransfusionRequirement.CreateDate] = trAntibody.CreateDate;
			drAntibody[TABLES.PatientTransfusionRequirement.CreateUser] = trAntibody.CreateUser;
			drAntibody[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = trAntibody.CreateDivisionCode;
			drAntibody[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = fromConversion;
			drAntibody[TABLES.PatientTransfusionRequirement.AntibodyTypeId] = 11;
			drAntibody[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drAntibody[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drAntibody[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtInsertTransfusionRequirements.Rows.Add(drAntibody);

			BOL.Patient.SaveSIsAndTRs(dtInsertSpecialInstructions, dtUpdateSpecialInstructions, dtInsertTransfusionRequirements, dtUpdateTransfusionRequirements, Common.UpdateFunction.UnitTests);
			UnitTestUtil.RunSQL("DELETE FROM BloodUnitAntigen WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'");
		}
		
		private void SetupAntigenTr(BOL.Patient patient, Guid bloodUnitGuid)
		{
			DataTable dtInsertSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(false);
			DataTable dtUpdateSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(true);
			DataTable dtInsertTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);
			DataTable dtUpdateTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(true);
					
			// Create new TR (antigen neg for C) 
			BOL.PatientTransfusionRequirement trAntigen = new BOL.PatientTransfusionRequirement(patient, "C", Common.TransfusionRequirementCategoryCode.AntigenNegative, UnitTestConfig.TestDivisionCode);
			DataRow drAntigen = dtInsertTransfusionRequirements.NewRow();

			Guid antigenGuid = Guid.NewGuid();
			drAntigen[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = antigenGuid;
			drAntigen[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drAntigen[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = trAntigen.TransfusionRequirement;
			drAntigen[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( trAntigen.TransfusionRequirementCategoryCode );
			drAntigen[TABLES.PatientTransfusionRequirement.CreateDate] = trAntigen.CreateDate;
			drAntigen[TABLES.PatientTransfusionRequirement.CreateUser] = trAntigen.CreateUser;
			drAntigen[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = trAntigen.CreateDivisionCode;
			drAntigen[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			drAntigen[TABLES.PatientTransfusionRequirement.AntigenTypeId] = 6;
			drAntigen[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drAntigen[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drAntigen[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtInsertTransfusionRequirements.Rows.Add(drAntigen);

			BOL.Patient.SaveSIsAndTRs(dtInsertSpecialInstructions, dtUpdateSpecialInstructions, dtInsertTransfusionRequirements, dtUpdateTransfusionRequirements, Common.UpdateFunction.UnitTests);
		}

		private void SetupAntibodyTr(BOL.Patient patient, Guid bloodUnitGuid)
		{
			DataTable dtInsertSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(false);
			DataTable dtUpdateSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(true);
			DataTable dtInsertTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);
			DataTable dtUpdateTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(true);
					
			// Create new TR (Anti-C antibody) 
			BOL.PatientTransfusionRequirement trAntibody = new BOL.PatientTransfusionRequirement(patient, "Anti-C", Common.TransfusionRequirementCategoryCode.AntibodyIdentified, UnitTestConfig.TestDivisionCode);
			DataRow drAntibody = dtInsertTransfusionRequirements.NewRow();

			Guid antibodyGuid = Guid.NewGuid();
			drAntibody[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = antibodyGuid;
			drAntibody[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drAntibody[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = trAntibody.TransfusionRequirement;
			drAntibody[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( trAntibody.TransfusionRequirementCategoryCode );
			drAntibody[TABLES.PatientTransfusionRequirement.CreateDate] = trAntibody.CreateDate;
			drAntibody[TABLES.PatientTransfusionRequirement.CreateUser] = trAntibody.CreateUser;
			drAntibody[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = trAntibody.CreateDivisionCode;
			drAntibody[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			drAntibody[TABLES.PatientTransfusionRequirement.AntibodyTypeId] = 5;
			drAntibody[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drAntibody[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drAntibody[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtInsertTransfusionRequirements.Rows.Add(drAntibody);

			BOL.Patient.SaveSIsAndTRs(dtInsertSpecialInstructions, dtUpdateSpecialInstructions, dtInsertTransfusionRequirements, dtUpdateTransfusionRequirements, Common.UpdateFunction.UnitTests);
		}



		public static BOL.BloodUnit CreateBloodUnit(string productTypeCode, 
			string productCode,
			Common.ABO abo,	
			Common.RH rH, 
			Common.ComponentClass componentClass
			)
		{
			return UnitTests.BOL_PatientUnitSelection.CreateBloodUnit(productTypeCode,
				productCode,
				abo, 
				rH, 
				componentClass, 
				UnitTestConfig.TestDivisionCode);
		}

		public static BOL.BloodUnit CreateBloodUnit(string productTypeCode, 
			string productCode,
			Common.ABO abo,	
			Common.RH rH, 
			Common.ComponentClass componentClass,
			string divisionCode)
		{
			// ensure unique unit id
			Random r = new Random();
			string unitId = "W" + r.Next(10000000,99999999).ToString() + "9999";

			BOL.BloodUnit bloodUnit = new BOL.BloodUnit(unitId, productCode, Common.BarcodeType.ISBT128, UnitTests.BOL_PatientUnitSelection.GetProductGuid(productCode));
			
			bloodUnit.BloodUnitMedia.AboRh.Abo = abo;
			bloodUnit.BloodUnitMedia.AboRh.RH = rH;
			bloodUnit.OriginalVolume = 100.0M;
			bloodUnit.BloodProductGuid = UnitTests.BOL_PatientUnitSelection.GetProductGuid(productCode);
			string sql = "SELECT TOP 1 IncomingShipmentGuid FROM BloodUnitShipment BUS JOIN BloodUnit BU ON BU.BloodUnitGuid = BUS.BloodUnitGuid AND BU.DivisionCode = '" + divisionCode + "' WHERE BUS.RecordStatusCode = 'A' AND BUS.IncomingShipmentGuid IS NOT NULL";
			DataTable dtTmp = UnitTestUtil.RunSQLGetDataTable(sql, false);
			if (dtTmp.Rows.Count == 0) DataCreator.CreateBloodUnit(false);		//Fix later, this is the lazy way
			DataRow dr = UnitTestUtil.RunSQLGetDataRow(sql);

			bloodUnit.IncomingShipmentGuid = (Guid) dr[TABLES.BloodUnitShipment.IncomingShipmentGuid];

			bloodUnit.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddDays(3);
			bloodUnit.DivisionCode					= divisionCode;
			bloodUnit.DonationType.DonationTypeCode	= Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.VolunteerAllogeneicDonor);
			bloodUnit.ProductCode = productCode;
			bloodUnit.ProductType.ProductTypeCode = productTypeCode;
			bloodUnit.BloodComponentClassId = (int) componentClass;

			UnitTests.BOL_PatientUnitSelection.InsertUnit(bloodUnit, bloodUnit.IncomingShipmentGuid);


			// Insert 4 tests (the random class is used to insure that the correct records are deleted at
			// the end of the test)
			string testResultID;
			//string entryTechID = random.Next(0,10000).ToString();
			for (int i=0; i<4; i++)
			{		
				int bloodTestTypeID;
				if (i<2) 
				{
					bloodTestTypeID = (int) Common.TestType.ABOInterp;
					testResultID = Common.Utility.GetAboFromEnum(bloodUnit.BloodUnitMedia.AboRh.Abo);
				}
				else 
				{
					bloodTestTypeID = (int) Common.TestType.RhInterp;
					testResultID = Common.Utility.GetRhFromEnum(bloodUnit.BloodUnitMedia.AboRh.RH);
				}
				sql = "INSERT INTO BloodUnitTest (TestDate, BloodUnitTestGuid, BloodUnitGuid, BloodTestTypeId, TestResultId, EntryTechID, EntryMethodCode, RecordStatusCode, DivisionCode, LastUpdateDate, LastUpdateUser) " + 
					"VALUES (getutcdate(), '" + Guid.NewGuid() + "','" + bloodUnit.BloodUnitGuid + "'," + bloodTestTypeID + ",'" + testResultID + "','" + "Foo" + "','M','A','" + divisionCode + "','" + DateTime.Now.ToString() + "','UnitTest')";

				UnitTestUtil.RunSQL(sql);
			}
			
			// bloodunit must be created again to get the new status
			return new BOL.BloodUnit(bloodUnit.BloodUnitGuid);
		}


		private static bool InsertUnit(BOL.BloodUnit bloodUnit, Guid incomingShipmentGuid)
		{
			//DataCreator.CreateBloodUnit(bloodUnit.BloodUnitGuid, bloodUnit.BloodUnitStatus.BloodUnitStatusGuid, System.Guid.Empty, bloodUnit.BloodProductGuid);
			DataCreator.CreateBloodUnit(bloodUnit.BloodUnitGuid, bloodUnit.BloodUnitStatus.BloodUnitStatusGuid, System.Guid.Empty, bloodUnit.BloodProductGuid, bloodUnit.BloodUnitMedia.AboRh.Abo, bloodUnit.BloodUnitMedia.AboRh.RH, bloodUnit.DivisionCode, (bloodUnit.LabelTypeCode == "C"));
			return true;


			//			DataTable dtBloodUnits = DAL.BloodUnitModification.GetEmptyBloodUnitTables();
			//			DataRow dr = dtBloodUnits.NewRow();
			//
			//			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);
			//			incomingShipmentGuid = bloodUnit.IncomingShipment.ShipmentGuid;
			//
			//			// Blood Unit
			//			dr[TABLES.BloodUnit.BloodUnitGuid]		= bloodUnit.BloodUnitGuid;
			//			dr[TABLES.BloodUnit.LabelUnitId]		= bloodUnit.EyeReadableUnitId;
			//			dr[TABLES.BloodUnit.EyeReadableUnitId] = bloodUnit.EyeReadableUnitId;
			//			dr[TABLES.BloodUnit.UnitProductCode]	= bloodUnit.ProductCode;
			//
			//			if (bloodUnit.ProductCode == string.Empty)
			//			{
			//				Assertion.Fail( "Console.WriteLine sometimes fails NUnit (during auto build) and this leads to ugly errors. Ask Stas for details if needed." );
			//				Console.WriteLine("Wrong!!!");
			//			}
			//			dr[TABLES.BloodUnitShipment.IncomingShipmentGuid] = bloodUnit.IncomingShipmentGuid;
			//			dr[TABLES.BloodUnit.DonationTypeId]	= bloodUnit.DonationType.DonationTypeId;
			//			dr[TABLES.BloodUnit.OriginalVolume]	= bloodUnit.OriginalVolume;
			//			dr[TABLES.BloodUnit.DivisionCode]		= bloodUnit.DivisionCode;
			//			dr[TABLES.BloodUnit.BloodProductGuid]	= bloodUnit.BloodProductGuid;
			//			dr[TABLES.BloodUnit.LabelTypeCode]		= bloodUnit.LabelTypeCode;
			//			dr[TABLES.BloodUnit.UnitDivisionId]	= (bloodUnit.LabelTypeCode == "C") ? string.Empty : "00";
			//
			//			dr[TABLES.BloodUnit.LastUpdateUser]	= "UnitTest";
			//
			//			dr[TABLES.BloodUnit.RecordStatusCode]	= Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			//			dr[TABLES.BloodUnit.LastUpdateFunctionId] = (int) Common.UpdateFunction.UnitTests;
			//
			//			
			//			// Blood Unit Media
			//			dr[TABLES.BloodUnitMedia.UnitExpirationDate]	= bloodUnit.BloodUnitMedia.UnitExpirationDate;
			//			dr[TABLES.BloodUnitMedia.BloodTypeCode]		= Common.Utility.GetAboFromEnum(bloodUnit.BloodUnitMedia.AboRh.Abo);
			//			dr[TABLES.BloodUnitMedia.RhFactorCode]			= Common.Utility.GetRhFromEnum(bloodUnit.BloodUnitMedia.AboRh.RH);
			//
			//			dr[TABLES.BloodUnitStatus.BloodUnitStatusGuid]				= bloodUnit.BloodUnitStatus.BloodUnitStatusGuid;
			//			dr[TABLES.BloodUnitStatus.BiohazardousWarningIndicator]	= bloodUnit.BloodUnitStatus.BiohazardousWarningIndicator;
			//
			//			if (((new StoredProcedure()).TransactionalGetValue(Common.VbecsStoredProcs.InsertMaintainableBloodUnit.StoredProcName, dr)) == 0)
			//			{
			//				//StringBuilder insert = new StringBuilder();
			//				
			//				string insert = "INSERT INTO BLOODUNITSHIPMENT (";
			//				insert += TABLES.BloodUnitShipment.BloodUnitShipmentGuid + ",";
			//				insert += TABLES.BloodUnitShipment.BloodUnitGuid + ",";
			//				insert += TABLES.BloodUnitShipment.IncomingShipmentGuid + ",";
			//				insert += TABLES.BloodUnitShipment.RecordStatusCode + ",";
			//				insert += TABLES.OrderedUnit.LastUpdateDate + ",";
			//				insert += TABLES.OrderedUnit.LastUpdateUser + ",";
			//				insert += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			//				insert += "VALUES (";
			//				insert += "'" + Guid.NewGuid() + "',";
			//				insert += "'" + bloodUnit.BloodUnitGuid + "',";
			//				insert += "'" + incomingShipmentGuid + "',";
			//				insert += "'A',";
			//				insert += "'" + Common.VBECSDateTime.FormatDateLongTimeString(DateTime.Now) + "',";
			//				insert += "'" + Environment.UserName + "',";
			//				insert += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//				insert += "\n";
			//				UnitTestUtil.RunSQL(insert);
			//				return true;
			//			}
			//			else
			//			{
			//				return false;
			//			}
		}



		private static Guid GetProductGuid(string productCode)
		{
			Guid productGuid = Guid.Empty;

			string sql = "SELECT * FROM BloodProduct WHERE ProductCode = '" + productCode.Substring(0,5) + "'";
			DataRow dr = UnitTestUtil.RunSQLGetDataRow(sql);
		
			if (!dr.IsNull(TABLES.BloodProduct.BloodProductGuid))
			{
				return (Guid) dr[TABLES.BloodProduct.BloodProductGuid];
			}
			else
			{
				return Guid.Empty;
			}
		}

		#endregion


		// unittests for the inner class
		#region selectedunits 

		[Test]
		public void SelectedUnits_Ctor_PatientUnitSelection_Pass()
		{
			BOL.PatientUnitSelection.SelectedUnits tmpSU = new BOL.PatientUnitSelection.SelectedUnits(this._patientUnitSelection);

			Assert.IsNotNull(tmpSU);
		}


		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SelectedUnits_Ctor_PatientUnitSelection_Fail()
		{
		}

		[Test]
		public void SelectedUnits_TempOrderedUnitComments_Pass()
		{
			DataTable tstValue = new DataTable();

			this._patientUnitSelection.SelectedUnitsCollection.TempOrderedUnitComments = tstValue;

			Assert.AreEqual(this._patientUnitSelection.SelectedUnitsCollection.TempOrderedUnitComments, tstValue);
		}


		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SelectedUnits_TempOrderedUnitComments_Fail()
		{
		}

		[Test]
		public void SelectedUnits_AlreadySelectedUnitsCrossmatchedSuccessfully_Pass()
		{
			int tstValue = new Random().Next(9999999);

			this._patientUnitSelection.SelectedUnitsCollection.AlreadySelectedUnitsCrossmatchedSuccessfully = tstValue;

			Assert.AreEqual(this._patientUnitSelection.SelectedUnitsCollection.AlreadySelectedUnitsCrossmatchedSuccessfully, tstValue);
		}


		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SelectedUnits_AlreadySelectedUnitsCrossmatchedSuccessfully_Fail()
		{
		}


		[Test]
		public void SelectedUnits_AddUnit_Pass()
		{
			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			BOL.BloodUnit bloodUnit = CreateBloodUnit("E002" ,"E0291100", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.RBC);
			bloodUnit.DonationType.DonationTypeCode = Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForAutologousUseOnly);
			bloodUnit.DirectedPatient = _patientUnitSelection.OrderedComponent.Patient;
			Guid guid = bloodUnit.BloodUnitGuid;
	

			Random random = new Random();
			string entryTechID = random.Next(0,10000).ToString();

			_patientUnitSelection.OrderedComponent.Patient.TransfusionRequirements.Rows.Clear();
			_patientUnitSelection.OrderedComponent.Patient.ClinicallySignificantAntibodies.Rows.Clear();
		
			_patientUnitSelection.SelectedUnitsCollection.AddUnit(bloodUnit, DateTime.Now, UnitTestConfig.User_Technologist, Common.UserRoleType.EnhancedSupervisor);
			Assert.IsTrue(this._patientUnitSelection.SelectedUnitsCollection.Count == 1);

			BOL.BloodUnit bloodUnit2 = this._patientUnitSelection.SelectedUnitsCollection.GetUnit(guid);
			Assert.IsTrue(bloodUnit2 != null && bloodUnit2.BloodUnitGuid == guid);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void SelectedUnits_AddUnit_Fail()
		{
			this._patientUnitSelection.SelectedUnitsCollection.AddUnit(null, System.DateTime.MinValue, UnitTestConfig.User_Technologist, Common.UserRoleType.Unknown);
		}


		[Test]
		public void SelectedUnits_RemoveUnit_Pass()
		{
			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			BOL.BloodUnit bloodUnit = CreateBloodUnit("E002" ,"E0291100", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.RBC);
			bloodUnit.DonationType.DonationTypeCode = Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForAutologousUseOnly);
			bloodUnit.DirectedPatient = _patientUnitSelection.OrderedComponent.Patient;
			Guid guid = bloodUnit.BloodUnitGuid;

			Random random = new Random();
			string entryTechID = random.Next(0,10000).ToString();

			_patientUnitSelection.OrderedComponent.Patient.TransfusionRequirements.Rows.Clear();
			_patientUnitSelection.OrderedComponent.Patient.ClinicallySignificantAntibodies.Rows.Clear();
		
			this._patientUnitSelection.SelectedUnitsCollection.AddUnit(bloodUnit, DateTime.Now, UnitTestConfig.User_Technologist, Common.UserRoleType.EnhancedSupervisor);
			
			if (this._patientUnitSelection.SelectedUnitsCollection.Count == 0)
			{
				Assert.Fail("The unit was not added");
			}
			int beforeCount = this._patientUnitSelection.SelectedUnitsCollection.Count;
			this._patientUnitSelection.SelectedUnitsCollection.RemoveUnit(guid);
			int afterCount = this._patientUnitSelection.SelectedUnitsCollection.Count;
			Assert.IsTrue(beforeCount == (afterCount + 1));
		}

		
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void SelectedUnits_RemoveUnit_Fail()
		{
			this._patientUnitSelection.SelectedUnitsCollection.RemoveUnit(System.Guid.Empty);
		}


		[Test]
		public void SelectedUnits_GetUnit_Pass()
		{
			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			BOL.BloodUnit bloodUnit = CreateBloodUnit("E002" ,"E0291100", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.RBC);
			bloodUnit.DonationType.DonationTypeCode = Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForAutologousUseOnly);
			bloodUnit.DirectedPatient = _patientUnitSelection.OrderedComponent.Patient;
			Guid guid = bloodUnit.BloodUnitGuid;

			Random random = new Random();
			string entryTechID = random.Next(0,10000).ToString();

			_patientUnitSelection.OrderedComponent.Patient.TransfusionRequirements.Rows.Clear();
			_patientUnitSelection.OrderedComponent.Patient.ClinicallySignificantAntibodies.Rows.Clear();
		
			_patientUnitSelection.SelectedUnitsCollection.AddUnit(bloodUnit, DateTime.Now, UnitTestConfig.User_Technologist, Common.UserRoleType.EnhancedSupervisor);	
			
			if (this._patientUnitSelection.SelectedUnitsCollection.Count == 0)
			{
				Assert.Fail("The unit was not added");
			}
			BOL.BloodUnit bloodUnit2 = this._patientUnitSelection.SelectedUnitsCollection.GetUnit(guid);
			Assert.IsTrue(bloodUnit2 != null);
			Assert.IsTrue(bloodUnit2.BloodUnitGuid == guid);
		}

				
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void SelectedUnits_GetUnit_Fail()
		{
			this._patientUnitSelection.SelectedUnitsCollection.GetUnit(System.Guid.Empty);
		}

		[Test]
		public void SelectedUnits_GetOrderedUnit_Pass()
		{
			_patientUnitSelection.OrderedComponent.Patient.AboRh.Abo = Common.ABO.O;
			_patientUnitSelection.OrderedComponent.Patient.AboRh.RH = Common.RH.Negative;

			BOL.BloodUnit bloodUnit = CreateBloodUnit("E002" ,"E0291100", Common.ABO.O, Common.RH.Negative, Common.ComponentClass.RBC);
			bloodUnit.DonationType.DonationTypeCode = Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForAutologousUseOnly);
			bloodUnit.DirectedPatient = _patientUnitSelection.OrderedComponent.Patient;
			Guid guid = bloodUnit.BloodUnitGuid;

			Random random = new Random();
			string entryTechID = random.Next(0,10000).ToString();

			_patientUnitSelection.OrderedComponent.Patient.TransfusionRequirements.Rows.Clear();
			_patientUnitSelection.OrderedComponent.Patient.ClinicallySignificantAntibodies.Rows.Clear();
		
			_patientUnitSelection.SelectedUnitsCollection.AddUnit(bloodUnit, DateTime.Now, UnitTestConfig.User_Technologist, Common.UserRoleType.EnhancedSupervisor);	
			
			if (_patientUnitSelection.SelectedUnitsCollection.Count == 0)
			{
				Assert.Fail("The unit was not added");
			}
			DataRow dr = _patientUnitSelection.SelectedUnitsCollection.GetOrderedUnit(guid);
			Assert.IsTrue(dr != null);
			Assert.IsTrue(( (Guid) dr[TABLES.OrderedUnit.OrderedComponentGuid] ) == this._orderedComponent.OrderedComponentGuid);
		}	
	
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void SelectedUnits_GetOrderedUnit_Fail()
		{
			_patientUnitSelection.SelectedUnitsCollection.GetOrderedUnit(System.Guid.Empty);
		}

		[Test]
		public void SelectedUnits_SaveUnits_Pass()
		{
			//this.RefreshData = true;

			// so it can be called from another UnitTest
			//if (this._patientUnitSelection == null) this.SetUp();

			BOL.BloodUnit bloodUnit = UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002" ,"E0291V00", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			// must do this to get RowVersion
			bloodUnit = new BOL.BloodUnit(bloodUnit.BloodUnitGuid);
			//			bloodUnit.BloodUnitStatus.UnitStatusCode = Common.UnitStatusCode.Assigned;
			_patientUnitSelection.SelectedUnitsCollection.AddUnit(bloodUnit, DateTime.Now, UnitTestConfig.User_Technologist, Common.UserRoleType.EnhancedSupervisor);
			Common.WorkloadProcessID workloadProcessID = Common.WorkloadProcessID.Unknown;
			Assert.IsTrue(_patientUnitSelection.SelectedUnitsCollection.SaveUnits(Common.UpdateFunction.UnitTests, new ArrayList(), ref workloadProcessID,null));
			UnitTestUtil.RunSQL("DELETE FROM OrderedUnit WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'");
		}
		

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SelectedUnits_SaveUnits_Fail()
		{
		}

		
		[Test]
		public void SelectedUnits_Count_Pass()
		{
			int tstValue = _patientUnitSelection.SelectedUnitsCollection.Count;
			Assert.AreEqual(_patientUnitSelection.SelectedUnitsCollection.Count, tstValue);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void SelectedUnits_Count_Fail()
		{
		}
		
		[Test]
		public void SelectedUnits_GetEnumerator_Pass()
		{
			System.Collections.IEnumerator tstValue = _patientUnitSelection.SelectedUnitsCollection.GetEnumerator();

			Assert.IsNotNull(tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SelectedUnits_GetEnumerator_Fail()
		{
		}
				
		[Test]
		public void SelectedUnits_MoveNext_Pass()
		{
			IEnumerator tst = this._patientUnitSelection.SelectedUnitsCollection.GetEnumerator();

			bool retValue = tst.MoveNext();

			Assert.IsNotNull(retValue);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SelectedUnits_MoveNext_Fail()
		{
		}


		[Test]
		public void SelectedUnits_AlreadySelectedUnits_Pass()
		{
			DataTable _AlreadySelectedUnits;
			_AlreadySelectedUnits = this._patientUnitSelection.SelectedUnitsCollection.AlreadySelectedUnits;
			Assert.AreEqual(this._patientUnitSelection.SelectedUnitsCollection.AlreadySelectedUnits,_AlreadySelectedUnits);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void SelectedUnits_AlreadySelectedUnits_Fail()
		{
		}

		[Test]
		public void SelectedUnits_DirectedUnits_Pass()
		{
			DataTable _DirectedUnits;
			_DirectedUnits = this._patientUnitSelection.SelectedUnitsCollection.DirectedUnits;
			Assert.AreEqual(this._patientUnitSelection.SelectedUnitsCollection.DirectedUnits,_DirectedUnits);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void SelectedUnits_DirectedUnits_Fail()
		{
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void SelectedUnits_Contains_Pass()
		{
			// For some reason this is not adding the blood unit. 
			// I couldn't tell you why.
			BOL.PatientUnitSelection.SelectedUnits tmpSUnits=new gov.va.med.vbecs.BOL.PatientUnitSelection.SelectedUnits(this._patientUnitSelection);
			
			tmpSUnits.AddUnit(_genericBU, DateTime.Today, UnitTestConfig.User_Technologist,Common.UserRoleType.Technologist);

			bool exists=tmpSUnits.Contains(_genericBU.BloodUnitGuid);
			
			Assert.IsTrue(exists);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void SelectedUnits_Contains_Fail()
		{
			BOL.PatientUnitSelection.SelectedUnits tmpSUnits=new gov.va.med.vbecs.BOL.PatientUnitSelection.SelectedUnits(this._patientUnitSelection);
			bool exists=tmpSUnits.Contains(System.Guid.Empty);

			Assert.IsTrue(!exists);
		}

		#endregion

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ClearClassObjects_Pass()
		{
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ClearClassObjects_Fail()
		{
		}

		[Test]
		public void AllAutologousUnitsChosen_Pass()
		{
			bool retValue = this._patientUnitSelection.AllAutologousUnitsChosen();
			Assert.AreEqual(true, retValue, "ReturnValue");
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void AllAutologousUnitsChosen_Fail()
		{
		}


		[Test]
		public void AllDirectedUnitsChosen_Pass()
		{
			bool retValue = this._patientUnitSelection.AllDirectedUnitsChosen();
			Assert.AreEqual(true, retValue, "ReturnValue");
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void AllDirectedUnitsChosen_Fail()
		{
		}


		[Test]
		public void AllRestrictedUnitsChosen_Pass()
		{
			bool retValue = this._patientUnitSelection.AllRestrictedUnitsChosen();
			Assert.AreEqual(true, retValue, "ReturnValue");
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void AllRestrictedUnitsChosen_Fail()
		{
		}


		[Test]
		public void AlreadySelectedInPreviousSession_Pass()
		{
			bool retValue = _patientUnitSelection.AlreadySelectedInPreviousSession(this._genericBU.BloodUnitGuid);
			Assert.AreEqual(false, retValue, "ReturnValue");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void AlreadySelectedInPreviousSession_Fail()
		{
		}

		// Begin SelectEventArgs
		[Test]
		public void SelectEventArgs_SelectEventArgs_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			Guid unitStatusGuid = Guid.NewGuid();
			gov.va.med.vbecs.BOL.PatientUnitSelection.SelectedUnits.SelectEventArgs sEA = new gov.va.med.vbecs.BOL.PatientUnitSelection.SelectedUnits.SelectEventArgs(unitGuid, unitStatusGuid);
			Assert.IsTrue(sEA.BloodUnitGuid==unitGuid && sEA.BloodUnitStatusGuid==unitStatusGuid);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SelectEventArgs_SelectEventArgs_Fail()
		{
		}


		[Test]
		public void SelectEventArgs_BloodUnitStatusGuid_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			Guid unitStatusGuid = Guid.NewGuid();
			gov.va.med.vbecs.BOL.PatientUnitSelection.SelectedUnits.SelectEventArgs sEA = new gov.va.med.vbecs.BOL.PatientUnitSelection.SelectedUnits.SelectEventArgs(unitGuid, unitStatusGuid);
			Assert.IsTrue(sEA.BloodUnitGuid==unitGuid);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SelectEventArgs_BloodUnitStatusGuid_Fail()
		{
		}


		[Test]
		public void SelectEventArgs_BloodUnitGuid_Pass()
		{
			Guid unitGuid = Guid.NewGuid();
			Guid unitStatusGuid = Guid.NewGuid();
			gov.va.med.vbecs.BOL.PatientUnitSelection.SelectedUnits.SelectEventArgs sEA = new gov.va.med.vbecs.BOL.PatientUnitSelection.SelectedUnits.SelectEventArgs(unitGuid, unitStatusGuid);
			Assert.IsTrue(sEA.BloodUnitStatusGuid==unitStatusGuid);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SelectEventArgs_BloodUnitGuid_Fail()
		{
		}

		// End SelectEventArgs

		[Test]
		public void AllDirectedUnitsChosen_BloodUnitGuid_Param_Pass()
		{
			bool retValue = this._patientUnitSelection.SelectedUnitsCollection.AllDirectedUnitsChosen(this._genericBU.BloodUnitGuid);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void AllDirectedUnitsChosen_BloodUnitGuid_Param_Fail()
		{
		}


		[Test]
		public void AllRestrictedUnitsChosen_BloodUnitGuid_Param_Pass()
		{
			bool retValue = this._patientUnitSelection.SelectedUnitsCollection.AllRestrictedUnitsChosen(this._genericBU.BloodUnitGuid);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void AllRestrictedUnitsChosen_BloodUnitGuid_Param_Fail()
		{
		}


		[Test]
		public void AllAutologousUnitsChosen_BloodUnitGuid_Param_Pass()
		{
			bool retValue = this._patientUnitSelection.SelectedUnitsCollection.AllAutologousUnitsChosen(this._genericBU.BloodUnitGuid);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void AllAutologousUnitsChosen_BloodUnitGuid_Param_Fail()
		{
		}

		// Begin ICollection 

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void CopyTo_Pass()
		{
			System.Array tstValue = (System.Array)"yadda".ToCharArray();

			this._patientUnitSelection.SelectedUnitsCollection.CopyTo(tstValue, 0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void CopyTo_Fail()
		{
		}


		[Test]
		public void IsSynchronized_Pass()
		{
			bool retValue = this._patientUnitSelection.SelectedUnitsCollection.IsSynchronized;

			Assert.AreEqual(false, retValue, "Worse property, EVER!");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void IsSynchronized_Fail()
		{
		}


		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SyncRoot_Pass()
		{
		}
		[Test]
		[ExpectedExceptionAttribute(typeof(BusinessObjectException))]
		public void SyncRoot_Fail()
		{
			Object foo = _patientUnitSelection.SelectedUnitsCollection.SyncRoot;
		}

		// End ICollection 

		// Begin IEnumerator 

		[Test]
		[Ignore(StandardIgnoreReasons.Untestable)]
		public void Reset_Pass()
		{
		}
		[Test]
		[Ignore(StandardIgnoreReasons.Untestable)]
		public void Reset_Fail()
		{
		}


		[Test]
		[Ignore(StandardIgnoreReasons.Untestable)]
		public void Current_Pass()
		{
		}

		[Test]
		[Ignore(StandardIgnoreReasons.Untestable)]
		public void Current_Fail()
		{
		}

		// End IEnumerator 
	}
}

#endif








